const fs = require('fs').promises;
const path = require('path');

// Menentukan path ke file database.json di direktori utama proyek
const dbPath = path.join(__dirname, '..', 'database.json');

/**
 * Membaca data dari database.json.
 * Jika file tidak ada, file baru akan dibuat dengan objek kosong.
 * **DIPERBARUI**: Menambahkan penanganan error untuk file JSON yang korup.
 * @returns {Promise<Object>} Data dari database.
 */
async function readDb() {
    try {
        const data = await fs.readFile(dbPath, 'utf8');
        // Coba parse JSON. Jika gagal, tangani di blok catch.
        return JSON.parse(data);
    } catch (error) {
        // Jika file tidak ada (ENOENT), buat file baru dan kembalikan objek kosong.
        if (error.code === 'ENOENT') {
            await writeDb({});
            return {};
        }
        // Jika terjadi error saat parsing (file korup), log error dan kembalikan objek kosong agar bot tidak crash.
        if (error instanceof SyntaxError) {
            console.error('FATAL ERROR: File database.json korup atau tidak valid. Mengembalikan database kosong untuk sesi ini.');
            console.error(`Detail Error: ${error.message}`);
            // Kembalikan objek kosong agar bot bisa terus berjalan tanpa data lama.
            // Anda harus memperbaiki file database.json secara manual.
            return {}; 
        }
        // Untuk error lainnya, tampilkan di console.
        console.error('Error saat membaca file database:', error);
        throw error;
    }
}

/**
 * Menulis data ke database.json.
 * @param {Object} data Objek yang akan ditulis ke file.
 */
async function writeDb(data) {
    try {
        await fs.writeFile(dbPath, JSON.stringify(data, null, 2), 'utf8');
    } catch (error) {
        console.error('Error saat menulis ke file database:', error);
        throw error;
    }
}

module.exports = { readDb, writeDb };
