const fs = require('fs');
const path = require('path');
const os = require('os'); // Diperlukan untuk mendapatkan home directory
const { exec } = require('child_process');
const crypto = require('crypto');

const dbPath = path.join(__dirname, '..', 'database.json');
const stateFilePath = path.join(__dirname, '..', 'rclone_backup_state.hash');

// Konfigurasi dari file .env
const remoteName = process.env.RCLONE_REMOTE_NAME || 'Backup';
const remotePath = process.env.RCLONE_REMOTE_PATH || 'DigitalOceanBot-Tester';

// --- FUNGSI BARU: Auto-install konfigurasi rclone ---
/**
 * Memastikan file konfigurasi rclone ada. Jika tidak, file akan dibuat
 * dengan data yang sama seperti pada skrip shell.
 */
function ensureRcloneConfig() {
    const homeDir = os.homedir();
    const rcloneConfigDir = path.join(homeDir, '.config', 'rclone');
    const rcloneConfigFile = path.join(rcloneConfigDir, 'rclone.conf');

    // Cek jika file sudah ada
    if (fs.existsSync(rcloneConfigFile)) {
        console.log('[Rclone Config] File rclone.conf sudah ada.');
        return;
    }

    console.log('[Rclone Config] File rclone.conf tidak ditemukan. Membuat konfigurasi baru...');

    // Konten file konfigurasi, sama persis seperti di auto-backup.sh
    const rcloneConfigContent = `[Backup]
type = drive
scope = drive
root_folder_id = 1qRyfhYB9TRTNSDinrkkTeyVoUUvZGG3K
token = {"access_token":"ya29.a0AS3H6Nyp3KVOZRPQeASwhwoCzFJntWEhi7E_DlBCSSt5QamS4wMqyl78OvhkH5YR1W2Z4IHR0RGgRC_AtzFR8e-7K556ASr9Yw-_qycKR2RXMEVEw33QdaSb1tvhQkgVgBWpiXkC-rbQYTPBPEUod6k6hqUyvckxHF3brA7qaCgYKARsSARQSFQHGX2Mio6iWZTAuPZwhh2K0zrTOdA0175","token_type":"Bearer","refresh_token":"1//0gnHr5v5lnUUZCgYIARAAGBASNwF-L9IrLEp43SYq3WESJExnG3ihKGuMuN7WDgEr9tFgBewPvTgJtaMlJ7Fg8gOEJqFJIXNXGDU","expiry":"2025-06-19T02:20:04.223867092+07:00"}
team_drive = 
`;

    try {
        // Buat direktori jika belum ada
        fs.mkdirSync(rcloneConfigDir, { recursive: true });
        // Tulis file konfigurasi
        fs.writeFileSync(rcloneConfigFile, rcloneConfigContent);
        console.log('[Rclone Config] File rclone.conf berhasil dibuat di:', rcloneConfigFile);
    } catch (error) {
        console.error('[Rclone Config] Gagal membuat file rclone.conf:', error);
    }
}
// --- AKHIR FUNGSI BARU ---


/**
 * Menjalankan perintah shell dan mengembalikannya sebagai Promise.
 * @param {string} command Perintah yang akan dijalankan.
 * @returns {Promise<string>} Output dari perintah.
 */
function runCommand(command) {
    return new Promise((resolve, reject) => {
        exec(command, (error, stdout, stderr) => {
            if (error) {
                if (stderr.includes('doesn\'t exist')) {
                    return resolve('');
                }
                return reject(new Error(stderr || error.message));
            }
            resolve(stdout.trim());
        });
    });
}

/**
 * Mendapatkan hash MD5 dari file lokal.
 * @returns {string|null} Hash MD5 atau null jika file tidak ada.
 */
function getLocalHash() {
    if (!fs.existsSync(dbPath)) return null;
    const fileBuffer = fs.readFileSync(dbPath);
    return crypto.createHash('md5').update(fileBuffer).digest('hex');
}

/**
 * Membaca hash terakhir yang disimpan dari file state.
 * @returns {string|null} Hash yang tersimpan atau null.
 */
function getLastKnownHash() {
    if (!fs.existsSync(stateFilePath)) return null;
    return fs.readFileSync(stateFilePath, 'utf8').trim();
}

/**
 * Menulis hash baru ke file state.
 * @param {string} hash Hash baru yang akan disimpan.
 */
function writeLastKnownHash(hash) {
    fs.writeFileSync(stateFilePath, hash, 'utf8');
}

/**
 * Fungsi utama yang menjalankan proses backup/sync dengan rclone.
 */
async function performRcloneBackup() {
    console.log(`[Rclone Backup] Memulai pengecekan backup pada ${new Date().toLocaleString('id-ID')}`);

    try {
        const localHash = getLocalHash();
        const lastKnownHash = getLastKnownHash();

        if (localHash && localHash !== lastKnownHash) {
            console.log('[Rclone Backup] Perubahan lokal terdeteksi. Mengunggah ke remote...');
            await runCommand(`rclone copy "${dbPath}" ${remoteName}:${remotePath}`);
            writeLastKnownHash(localHash);
            console.log('[Rclone Backup] Unggah berhasil.');
        } else {
            console.log('[Rclone Backup] Tidak ada perubahan lokal. Mengecek remote...');
            const remoteHashOutput = await runCommand(`rclone md5sum ${remoteName}:${remotePath}/database.json`);
            const remoteHash = remoteHashOutput.split(' ')[0];

            if (remoteHash && remoteHash !== localHash) {
                console.log('[Rclone Backup] Perubahan remote terdeteksi. Mengunduh dari remote...');
                await runCommand(`rclone copy ${remoteName}:${remotePath}/database.json "${path.dirname(dbPath)}"`);
                const newLocalHash = getLocalHash();
                writeLastKnownHash(newLocalHash);
                console.log('[Rclone Backup] Unduh berhasil.');
            } else {
                console.log('[Rclone Backup] Lokal dan remote sudah sinkron.');
            }
        }
    } catch (error) {
        console.error('[Rclone Backup] Terjadi kesalahan:', error.message);
    }
}

/**
 * Memulai jadwal backup otomatis.
 * @param {number} intervalMinutes Interval dalam menit.
 */
function startAutoBackup(intervalMinutes = 10) {
    // --- PERUBAHAN DIMULAI: Panggil fungsi auto-install ---
    ensureRcloneConfig();
    // --- PERUBAHAN SELESAI ---
    
    console.log(`[Rclone Backup] Auto-backup via rclone aktif, berjalan setiap ${intervalMinutes} menit.`);
    
    performRcloneBackup();

    setInterval(performRcloneBackup, intervalMinutes * 60 * 1000);
}

module.exports = { startAutoBackup };
