const fs = require('fs').promises;
const path = require('path');
const USERS_DB_PATH = path.join(__dirname, '..', 'users.json');

/**
 * Membaca database pengguna dari users.json.
 * @returns {Promise<Array<Object|number>>} Array berisi data pengguna.
 */
async function readUsersDb() {
    try {
        const data = await fs.readFile(USERS_DB_PATH, 'utf8');
        const users = JSON.parse(data);
        // Handle case where file might be empty or not an array
        if (!Array.isArray(users)) {
            await fs.writeFile(USERS_DB_PATH, JSON.stringify([], null, 2));
            return [];
        }
        return users;
    } catch (error) {
        if (error.code === 'ENOENT') {
            await fs.writeFile(USERS_DB_PATH, JSON.stringify([], null, 2));
            return [];
        }
        console.error('Error reading users database:', error);
        return [];
    }
}

/**
 * Menambahkan atau memperbarui pengguna di database.
 * @param {number} chatId ID chat pengguna.
 * @param {string|null} username Username pengguna.
 */
async function addUser(chatId, username) {
    let users = await readUsersDb();
    const numericChatId = Number(chatId);

    // Cari indeks pengguna berdasarkan ID
    const userIndex = users.findIndex(user => {
        const id = typeof user === 'object' ? user.id : user;
        return id === numericChatId;
    });

    if (userIndex !== -1) {
        // Pengguna sudah ada, perbarui datanya
        const existingUser = users[userIndex];
        // Hanya perbarui jika formatnya sudah objek atau jika username berubah
        if (typeof existingUser === 'object') {
            if (existingUser.username !== username) {
                existingUser.username = username;
                console.log(`[User DB] Updated username for ${numericChatId} to ${username}`);
            }
        } else {
            // Ubah dari format lama (hanya ID) ke format baru (objek)
            users[userIndex] = { id: numericChatId, username: username };
            console.log(`[User DB] Converted user ${numericChatId} to new format.`);
        }
    } else {
        // Pengguna baru, tambahkan dengan format baru
        users.push({ id: numericChatId, username: username });
        console.log(`[User DB] New user added: ${numericChatId}. Total users: ${users.length}`);
    }
    
    // Tulis kembali seluruh database yang sudah diperbarui
    await fs.writeFile(USERS_DB_PATH, JSON.stringify(users, null, 2));
}


/**
 * Mencari pengguna berdasarkan username (case-insensitive).
 * @param {string} username Username yang akan dicari.
 * @returns {Promise<Object|null>} Objek pengguna jika ditemukan, null jika tidak.
 */
async function findUserByUsername(username) {
    const users = await readUsersDb();
    const lowerCaseUsername = username.toLowerCase();
    
    // Hanya cari di entri yang sudah berformat objek dan memiliki username
    const foundUser = users.find(user => 
        typeof user === 'object' && 
        user.username && 
        user.username.toLowerCase() === lowerCaseUsername
    );

    return foundUser || null;
}

module.exports = { readUsersDb, addUser, findUserByUsername };

