const { readUsersDb } = require('../config/users');

const broadcastState = {};

/**
 * Memulai proses broadcast (dipicu oleh command atau tombol).
 * @param {import('node-telegram-bot-api')} bot
 * @param {import('node-telegram-bot-api').Message} msg
 */
async function handleBroadcastCommand(bot, msg) {
    const chatId = msg.chat.id;
    broadcastState[chatId] = { step: 'ask_message' };
    await bot.sendMessage(chatId, 'Silakan ketik pesan yang ingin Anda kirim ke semua pengguna.. Kirim /batal untuk membatalkan.');
}

/**
 * Menangani input teks dari admin selama proses broadcast.
 * @param {import('node-telegram-bot-api')} bot
 * @param {import('node-telegram-bot-api').Message} msg
 * @returns {Promise<boolean>}
 */
async function handleTextMessage(bot, msg) {
    const chatId = msg.chat.id;
    const text = msg.text;

    if (!broadcastState[chatId] || broadcastState[chatId].step !== 'ask_message') {
        return false;
    }

    if (text === '/batal') {
        delete broadcastState[chatId];
        await bot.sendMessage(chatId, 'Broadcast dibatalkan.');
        return true;
    }

    const state = broadcastState[chatId];
    state.message = text;
    state.step = 'confirm';

    const users = await readUsersDb();
    const userCount = users.length;

    let confirmationText = `<b>KONFIRMASI BROADCAST</b>\n\n`;
    confirmationText += `Pesan berikut akan dikirim ke <b>${userCount} pengguna</b>:\n\n`;
    confirmationText += `------------------------------------\n`;
    confirmationText += `${text}\n`;
    confirmationText += `------------------------------------\n\n`;
    confirmationText += `Apakah Anda yakin ingin mengirim pesan ini?`;

    const keyboard = [
        [{ text: '✅ Ya, Kirim Sekarang', callback_data: 'broadcast_confirm_yes' }],
        [{ text: '⛔ Tidak, Batalkan', callback_data: 'broadcast_cancel' }]
    ];

    await bot.sendMessage(chatId, confirmationText, { 
        parse_mode: 'HTML', 
        reply_markup: { inline_keyboard: keyboard } 
    });

    return true;
}

/**
 * Menangani tombol callback untuk konfirmasi atau pembatalan broadcast.
 * @param {import('node-telegram-bot-api')} bot
 * @param {import('node-telegram-bot-api').CallbackQuery} cbq
 * @returns {Promise<boolean>}
 */
async function handleCallbackQuery(bot, cbq) {
    const data = cbq.data;
    if (!data.startsWith('broadcast_')) return false;

    const msg = cbq.message;
    const chatId = msg.chat.id;

    await bot.deleteMessage(chatId, msg.message_id).catch(() => {});

    if (data === 'broadcast_cancel') {
        delete broadcastState[chatId];
        await bot.sendMessage(chatId, 'Broadcast dibatalkan.');
        return true;
    }

    const state = broadcastState[chatId];
    if (!state || state.step !== 'confirm') return true;

    if (data === 'broadcast_confirm_yes') {
        const messageToSend = state.message;
        delete broadcastState[chatId];

        const users = await readUsersDb();
        
        // --- PERUBAHAN DI SINI ---
        // Menambahkan jumlah pengguna pada pesan "memulai broadcast"
        await bot.sendMessage(chatId, `🚀 Memulai broadcast ke <b>${users.length} pengguna</b>... Harap tunggu laporan selesai.`, { parse_mode: 'HTML' });
        
        let successCount = 0;
        let failureCount = 0;

        // Loop melalui objek pengguna dan ambil ID-nya
        for (const user of users) {
            // Memastikan kita selalu memiliki ID, baik dari format lama maupun baru
            const targetId = typeof user === 'object' ? user.id : user;
            if (!targetId) continue;

            try {
                await bot.sendMessage(targetId, messageToSend, { parse_mode: 'HTML' });
                successCount++;
            } catch (error) {
                console.error(`Gagal mengirim broadcast ke ${targetId}:`, error.response ? error.response.body.description : error.message);
                failureCount++;
            }
            // Memberi jeda kecil untuk menghindari limitasi Telegram
            await new Promise(resolve => setTimeout(resolve, 100)); 
        }

        let reportText = `<b>📊 Laporan Broadcast Selesai</b>\n\n`;
        reportText += `✅ Berhasil terkirim: ${successCount} pengguna\n`;
        reportText += `❌ Gagal terkirim: ${failureCount} pengguna`;

        await bot.sendMessage(chatId, reportText, { parse_mode: 'HTML' });
    }

    return true;
}

module.exports = {
    handleBroadcastCommand,
    handleTextMessage,
    handleCallbackQuery
};

