const axios = require('axios');

const API_URL = 'https://api.digitalocean.com/v2';

const getAxiosConfig = (token) => ({
    headers: { 'Authorization': `Bearer ${token}`, 'Content-Type': 'application/json' },
    timeout: 15000
});

const handleError = (error) => {
    if (error.response) {
        return { success: false, status: error.response.status, message: error.response.data.message || 'Unknown API error' };
    }
    return { success: false, message: error.message };
};

// === FUNGSI AKUN ===
async function getAccountStatus(token) {
    try {
        const response = await axios.get(`${API_URL}/account`, getAxiosConfig(token));
        return { success: true, data: response.data.account };
    } catch (error) {
        return handleError(error);
    }
}

/**
 * FUNGSI DIPERBARUI: Mendapatkan informasi saldo dengan logika yang lebih tangguh.
 * @param {string} token - Token API DigitalOcean.
 * @returns {Promise<object>} Hasil yang berisi data saldo.
 */
async function getAccountBalance(token) {
    let balanceData = null;
    let isNewAccountScenario = false;

    try {
        // Prioritas 1: Coba ambil saldo langsung dari endpoint /balance
        const balanceResponse = await axios.get(`${API_URL}/customers/my/balance`, getAxiosConfig(token));
        balanceData = balanceResponse.data;
        console.log('[Balance Check] Berhasil mendapatkan saldo dari /balance.');

        // Jika saldo $0.00, ini bisa jadi akun baru. Tandai untuk pemeriksaan lebih lanjut.
        // PERUBAHAN: Memeriksa apakah saldo kurang dari $10
        if (parseFloat(balanceData.account_balance) > -10) {
            console.log('[Balance Check] Saldo terdeteksi kecil, mengasumsikan akun baru.');
            isNewAccountScenario = true;
        } else {
             // Jika saldo tidak nol, langsung kembalikan hasilnya.
             return { success: true, data: balanceData };
        }

    } catch (error) {
        // Prioritas 2: Jika gagal dengan error 404, ini pasti akun baru
        if (error.response && error.response.status === 404) {
            console.log('[Balance Check] Gagal mendapatkan saldo (404), ini adalah akun baru.');
            isNewAccountScenario = true;
        } else {
            // Untuk error lain (misal: 401 Unauthorized), laporkan sebagai kegagalan
            return handleError(error);
        }
    }

    // Jika ini adalah skenario akun baru (baik dari 404 atau saldo $0), periksa /credits
    if (isNewAccountScenario) {
        try {
            console.log('[Balance Check] Memeriksa /credits untuk akun baru...');
            const creditsResponse = await axios.get(`${API_URL}/customers/my/credits`, getAxiosConfig(token));
            const credits = creditsResponse.data.credits;
            
            let totalRemainingCredits = 0;
            if (credits && credits.length > 0) {
                totalRemainingCredits = credits.reduce((sum, credit) => sum + parseFloat(credit.amount), 0);
            }

            console.log(`[Balance Check] Ditemukan total kredit free trial: $${totalRemainingCredits}`);

            // Jika ada kredit, gunakan nilai ini. Jika tidak, gunakan nilai $200
            if (totalRemainingCredits > 0) {
                return {
                    success: true,
                    data: {
                        account_balance: `-${totalRemainingCredits.toFixed(2)}`,
                        month_to_date_balance: '0.00'
                    }
                };
            } else {
                 // PERUBAHAN: Jika tidak ada kredit, kembalikan saldo $200
                 return { success: true, data: { account_balance: '-200.00', month_to_date_balance: '0.00' } };
            }
        } catch (creditError) {
            // Jika /credits juga gagal, gunakan asumsi default sebagai upaya terakhir
            console.log('[Balance Check] Gagal memeriksa /credits, mengasumsikan saldo default $200.');
            return {
                success: true,
                data: {
                    account_balance: '-200.00',
                    month_to_date_balance: '0.00'
                }
            };
        }
    }
}


async function getDropletCount(token) {
    try {
        const response = await axios.get(`${API_URL}/droplets?per_page=1`, getAxiosConfig(token));
        return { success: true, total: response.data.meta.total };
    } catch (error) {
        return handleError(error);
    }
}

// === FUNGSI DROPLET ===
async function getDroplets(token) {
    try {
        let droplets = [];
        let url = `${API_URL}/droplets?per_page=200`;
        while (url) {
            const response = await axios.get(url, getAxiosConfig(token));
            droplets = droplets.concat(response.data.droplets);
            url = response.data.links?.pages?.next || null;
        }
        return { success: true, data: droplets };
    } catch (error) {
        return handleError(error);
    }
}

async function getDropletDetails(token, dropletId) {
    try {
        const response = await axios.get(`${API_URL}/droplets/${dropletId}`, getAxiosConfig(token));
        return { success: true, data: response.data };
    } catch (error) {
        return handleError(error);
    }
}

async function createDroplet(token, options) {
    try {
        const response = await axios.post(`${API_URL}/droplets`, options, getAxiosConfig(token));
        return { success: true, data: response.data };
    } catch (error) {
        return handleError(error);
    }
}

async function deleteDroplet(token, dropletId) {
    try {
        await axios.delete(`${API_URL}/droplets/${dropletId}`, getAxiosConfig(token));
        return { success: true };
    } catch (error) {
        return handleError(error);
    }
}

async function rebootDroplet(token, dropletId) {
    try {
        const payload = { type: 'reboot' };
        const response = await axios.post(`${API_URL}/droplets/${dropletId}/actions`, payload, getAxiosConfig(token));
        return { success: true, data: response.data.action };
    } catch (error) {
        return handleError(error);
    }
}

// === FUNGSI DATA PENDUKUNG ===
async function getRegions(token) {
    try {
        const response = await axios.get(`${API_URL}/regions?per_page=200`, getAxiosConfig(token));
        const availableRegions = response.data.regions.filter(r => r.available);
        return { success: true, data: availableRegions };
    } catch (error) {
        return handleError(error);
    }
}

async function getSizes(token) {
     try {
        const response = await axios.get(`${API_URL}/sizes?per_page=200`, getAxiosConfig(token));
        return { success: true, data: response.data.sizes };
    } catch (error) {
        return handleError(error);
    }
}

async function getImages(token) {
     try {
        const response = await axios.get(`${API_URL}/images?type=distribution&per_page=200`, getAxiosConfig(token));
        const availableImages = response.data.images.filter(img => img.status === 'available');
        return { success: true, data: availableImages };
    } catch (error) {
        return handleError(error);
    }
}

// Pastikan semua fungsi diekspor dengan benar
module.exports = {
    getAccountStatus,
    getAccountBalance,
    getDropletCount,
    getDroplets,
    getDropletDetails,
    createDroplet,
    deleteDroplet,
    rebootDroplet,
    getRegions,
    getSizes,
    getImages
};
