const { google } = require('googleapis');
const path = require('path');
const fs = require('fs');
require('dotenv').config();

const CREDENTIALS_PATH = path.join(process.cwd(), 'credentials.json');

/**
 * Mendapatkan nama sheet target berdasarkan bulan dan tahun saat ini.
 * @returns {string} Nama sheet, contoh: "September 2025".
 */
function getCurrentSheetName() {
    const now = new Date();
    // Menggunakan locale 'id-ID' untuk nama bulan dalam Bahasa Indonesia
    const month = now.toLocaleString('id-ID', { month: 'long', timeZone: 'Asia/Jakarta' });
    const year = now.getFullYear();
    return `${month} ${year}`;
}


/**
 * Mengautentikasi dan mengembalikan klien Google Sheets yang siap digunakan.
 * @returns {Promise<{sheets: any, auth: any}>} Objek yang berisi instance sheets API dan auth client.
 */
async function getGoogleSheetsClient() {
    if (!fs.existsSync(CREDENTIALS_PATH)) {
        throw new Error("File credentials.json tidak ditemukan. Harap ikuti panduan di README_SPREADSHEET.md.");
    }
    const auth = new google.auth.GoogleAuth({
        keyFile: CREDENTIALS_PATH,
        scopes: 'https://www.googleapis.com/auth/spreadsheets',
    });
    const authClient = await auth.getClient();
    const sheets = google.sheets({ version: 'v4', auth: authClient });
    return { sheets, auth };
}

/**
 * Menambahkan baris data baru ke sheet bulanan yang sesuai.
 * Jika sheet untuk bulan ini belum ada, sheet akan dibuat secara otomatis.
 * @param {Array<string>} rowData Array nilai yang akan ditambahkan sebagai satu baris.
 */
async function appendRow(rowData) {
    const GOOGLE_SHEET_ID = process.env.GOOGLE_SHEET_ID;
    if (!GOOGLE_SHEET_ID) {
        console.error('[Google Sheets] Error: GOOGLE_SHEET_ID tidak diatur di file .env.');
        return;
    }

    try {
        const { sheets } = await getGoogleSheetsClient();
        const sheetName = getCurrentSheetName();

        // Cek apakah sheet untuk bulan ini sudah ada
        const spreadsheetMeta = await sheets.spreadsheets.get({ spreadsheetId: GOOGLE_SHEET_ID });
        const sheetExists = spreadsheetMeta.data.sheets.some(s => s.properties.title === sheetName);

        // Jika sheet belum ada, buat sheet baru dan tambahkan header
        if (!sheetExists) {
            console.log(`[Google Sheets] Sheet "${sheetName}" tidak ditemukan, membuat baru...`);
            // 1. Buat sheet baru
            await sheets.spreadsheets.batchUpdate({
                spreadsheetId: GOOGLE_SHEET_ID,
                requestBody: {
                    requests: [{ addSheet: { properties: { title: sheetName } } }],
                },
            });

            // 2. Tambahkan header ke sheet yang baru dibuat
            const headerRow = [[
                'Timestamp', 'Jenis', 'Hostname', 'Ukuran', 'Region', 'IP Address', 'OS', 'Password', 'Kedaluwarsa', 'Akun API', 'User ID Pembeli', 'Status'
            ]];
            await sheets.spreadsheets.values.append({
                spreadsheetId: GOOGLE_SHEET_ID,
                range: `${sheetName}!A1`,
                valueInputOption: 'USER_ENTERED',
                requestBody: {
                    values: headerRow,
                },
            });
             console.log(`[Google Sheets] Sheet "${sheetName}" berhasil dibuat beserta headernya.`);
        }

        // Tambahkan baris data ke sheet yang sesuai
        await sheets.spreadsheets.values.append({
            spreadsheetId: GOOGLE_SHEET_ID,
            range: sheetName,
            valueInputOption: 'USER_ENTERED',
            requestBody: {
                values: [rowData],
            },
        });
        console.log(`[Google Sheets] Baris baru berhasil ditambahkan ke sheet "${sheetName}".`);
    } catch (error) {
        console.error('[Google Sheets] Gagal menambahkan baris:', error.message);
    }
}

/**
 * Mencari baris berdasarkan alamat IP di semua sheet, lalu mengubah statusnya menjadi "Dihapus"
 * dan mewarnai baris tersebut.
 * @param {string} ipAddress Alamat IP yang akan dicari.
 */
async function updateRowStatusToDeleted(ipAddress) {
    const GOOGLE_SHEET_ID = process.env.GOOGLE_SHEET_ID;
    if (!GOOGLE_SHEET_ID) {
        console.error('[Google Sheets] Error: GOOGLE_SHEET_ID tidak diatur di file .env.');
        return;
    }
    if (!ipAddress) {
         console.error('[Google Sheets] Peringatan: Upaya memperbarui status tanpa IP Address.');
         return;
    }


    try {
        const { sheets } = await getGoogleSheetsClient();

        // 1. Dapatkan semua sheet yang ada di dalam spreadsheet
        const spreadsheetMeta = await sheets.spreadsheets.get({ spreadsheetId: GOOGLE_SHEET_ID });
        const allSheets = spreadsheetMeta.data.sheets;
        
        // 2. Iterasi setiap sheet untuk mencari IP
        for (const sheet of allSheets) {
            const sheetTitle = sheet.properties.title;
            const sheetId = sheet.properties.sheetId;

            const response = await sheets.spreadsheets.values.get({
                spreadsheetId: GOOGLE_SHEET_ID,
                range: sheetTitle,
            });

            const rows = response.data.values;
            if (!rows || rows.length === 0) continue; // Lanjut ke sheet berikutnya jika kosong

            const header = rows[0];
            const ipColumnIndex = header.indexOf('IP Address');
            const statusColumnIndex = header.indexOf('Status');

            if (ipColumnIndex === -1 || statusColumnIndex === -1) {
                console.error(`[Google Sheets] Header 'IP Address' atau 'Status' tidak ditemukan di sheet "${sheetTitle}".`);
                continue; // Lanjut ke sheet berikutnya
            }

            for (let i = 1; i < rows.length; i++) {
                const row = rows[i];
                if (row[ipColumnIndex] === ipAddress) {
                    const rowIndex = i; // Index baris (dimulai dari 0)
                    
                    // 3. Update status menjadi "Dihapus"
                    await sheets.spreadsheets.values.update({
                        spreadsheetId: GOOGLE_SHEET_ID,
                        range: `${sheetTitle}!${String.fromCharCode(65 + statusColumnIndex)}${rowIndex + 1}`,
                        valueInputOption: 'USER_ENTERED',
                        requestBody: {
                            values: [['Dihapus']],
                        },
                    });

                    // 4. Warnai baris
                    await sheets.spreadsheets.batchUpdate({
                        spreadsheetId: GOOGLE_SHEET_ID,
                        requestBody: {
                            requests: [{
                                updateCells: {
                                    range: {
                                        sheetId: sheetId,
                                        startRowIndex: rowIndex,
                                        endRowIndex: rowIndex + 1,
                                    },
                                    rows: [{
                                        values: [{
                                            userEnteredFormat: {
                                                backgroundColor: { red: 1.0, green: 0.8, blue: 0.8 } // Merah muda
                                            }
                                        }]
                                    }],
                                    fields: 'userEnteredFormat.backgroundColor'
                                }
                            }]
                        }
                    });

                    console.log(`[Google Sheets] Status untuk IP ${ipAddress} di sheet "${sheetTitle}" berhasil diubah menjadi Dihapus.`);
                    return; // Hentikan pencarian setelah ditemukan dan diupdate
                }
            }
        }
        
        console.log(`[Google Sheets] IP ${ipAddress} tidak ditemukan di sheet manapun.`);

    } catch (error) {
        console.error(`[Google Sheets] Gagal memperbarui status baris untuk IP ${ipAddress}:`, error.message);
    }
}


/**
 * Menguji koneksi ke Google Sheets dengan membaca judul spreadsheet.
 * @returns {Promise<{success: boolean, message: string}>} Hasil tes koneksi.
 */
async function testConnection() {
    try {
        const GOOGLE_SHEET_ID = process.env.GOOGLE_SHEET_ID;
        if (!GOOGLE_SHEET_ID) {
            return { success: false, message: 'GOOGLE_SHEET_ID tidak diatur di file .env.' };
        }
        
        const { sheets } = await getGoogleSheetsClient();
        const response = await sheets.spreadsheets.get({
            spreadsheetId: GOOGLE_SHEET_ID,
        });

        const title = response.data.properties.title;
        return { success: true, message: `Berhasil terhubung ke spreadsheet dengan judul: "${title}"` };

    } catch (error) {
        let detail = error.message;
        if (error.response && error.response.data && error.response.data.error) {
            detail = error.response.data.error.message;
        }
        return { success: false, message: `Koneksi Gagal: ${detail}` };
    }
}


module.exports = {
    appendRow,
    updateRowStatusToDeleted,
    testConnection
};

