const axios = require('axios');
const crypto = require('crypto');

// Ambil kredensial dari environment variables
const TRIPAY_API_KEY = process.env.TRIPAY_API_KEY;
const TRIPAY_PRIVATE_KEY = process.env.TRIPAY_PRIVATE_KEY;
const TRIPAY_MERCHANT_CODE = process.env.TRIPAY_MERCHANT_CODE;
const TRIPAY_API_URL = process.env.TRIPAY_API_URL || 'https://tripay.co.id/api-sandbox'; // Gunakan sandbox untuk development

/**
 * Membuat signature yang diperlukan untuk otentikasi API Tripay.
 * @param {string} merchantRef ID referensi unik dari merchant.
 * @param {number} amount Jumlah total pembayaran.
 * @returns {string} Signature yang telah di-generate.
 */
function createSignature(merchantRef, amount) {
    const data = `${TRIPAY_MERCHANT_CODE}${merchantRef}${amount}`;
    return crypto.createHmac('sha256', TRIPAY_PRIVATE_KEY).update(data).digest('hex');
}

/**
 * Meminta pembuatan transaksi baru ke Tripay.
 * @param {number} amount Jumlah pembayaran.
 * @param {string} method Kode channel pembayaran (e.g., 'QRIS', 'BCAVA').
 * @param {object} customer Data pelanggan { name, email, phone }.
 * @param {Array<object>} orderItems Daftar item pesanan.
 * @param {string} returnUrl URL untuk redirect setelah pembayaran.
 * @returns {Promise<object>} Hasil dari permintaan transaksi.
 */
async function requestTransaction(amount, method, customer, orderItems) {
    // Membuat ID unik untuk setiap transaksi
    const merchantRef = `DO-${Date.now()}`;
    const signature = createSignature(merchantRef, amount);

    const payload = {
        method,
        merchant_ref: merchantRef,
        amount,
        customer_name: customer.name,
        customer_email: customer.email,
        customer_phone: customer.phone,
        order_items: orderItems,
        // URL callback tidak kita gunakan di bot, tapi wajib diisi
        callback_url: 'https://tokojk.com/tripay_callback.php',
        return_url: 'https://tokojk.com/akun.html',
        expired_time: (Math.floor(Date.now() / 1000)) + (24 * 60 * 60), // Expired dalam 24 jam
        signature
    };

    try {
        const response = await axios.post(`${TRIPAY_API_URL}/transaction/create`, payload, {
            headers: { 'Authorization': `Bearer ${TRIPAY_API_KEY}` }
        });
        return response.data;
    } catch (error) {
        console.error('Tripay API Error:', error.response ? error.response.data : error.message);
        return { success: false, message: error.response?.data?.message || 'Gagal membuat transaksi Tripay.' };
    }
}

/**
 * Memeriksa detail atau status sebuah transaksi di Tripay.
 * @param {string} reference Kode referensi transaksi dari Tripay.
 * @returns {Promise<object>} Detail transaksi.
 */
async function checkTransactionStatus(reference) {
    if (!reference) {
        return { success: false, message: 'Referensi tidak valid.' };
    }
    try {
        const response = await axios.get(`${TRIPAY_API_URL}/transaction/detail?reference=${reference}`, {
            headers: { 'Authorization': `Bearer ${TRIPAY_API_KEY}` }
        });
        return response.data;
    } catch (error) {
        console.error('Tripay Check Status Error:', error.response ? error.response.data : error.message);
        return { success: false, message: 'Gagal memeriksa status transaksi.' };
    }
}


module.exports = {
    requestTransaction,
    checkTransactionStatus
};
