/**
 * Script Node.js untuk mengirim pesan ke beberapa grup Telegram secara terjadwal.
 * Script ini menggunakan pustaka 'telegram' untuk berinteraksi dengan API Telegram
 * dan 'node-cron' untuk penjadwalan.
 *
 * Fitur Utama:
 * 1. Login Otomatis: Sesi login disimpan dalam file, jadi Anda hanya perlu
 * memasukkan nomor telepon & kode verifikasi sekali saja.
 * 2. Mengirim ke Banyak Grup: Daftar grup target dibaca dari file `grup.txt`.
 * 3. Terjadwal: Pesan dikirim secara otomatis setiap 2 jam sekali.
 * 4. Pengiriman Instan: Pesan juga dikirim satu kali saat skrip pertama kali dijalankan.
 * 5. Menyalin Pesan (Copy-Paste): Mengambil konten pesan spesifik dan mengirimkannya
 * sebagai pesan baru, sehingga tidak terlihat seperti pesan terusan (forward).
 * 6. Mendukung Username & ID: Dapat mengirim pesan ke grup menggunakan username (teks)
 * maupun ID grup (angka).
 *
 * Dibuat oleh: Gemini
 * Tanggal: 28 Juli 2025
 */

// Impor pustaka yang diperlukan
const { TelegramClient } = require("telegram");
const { StringSession } = require("telegram/sessions");
const input = require("input"); // Untuk menangani input dari terminal
const fs = require("fs"); // Untuk membaca dan menulis file
const cron = require("node-cron"); // Untuk penjadwalan tugas

// --- KONFIGURASI ---
// ❗️ Ganti dengan apiId dan apiHash Anda sendiri yang didapat dari my.telegram.org
const apiId = 2316947; // Ganti dengan API ID Anda
const apiHash = "0967acc6bc907f7f694c4ffa8a8e3175"; // Ganti dengan API Hash Anda

// Lokasi file untuk menyimpan sesi agar tidak perlu login berulang kali
const SESSION_FILE_PATH = "./my_session.txt";

// Lokasi file yang berisi daftar grup (username atau ID)
const GRUP_FILE = "grup.txt";

// Tentukan ID channel dan ID pesan yang kontennya akan disalin
// Berdasarkan link: https://t.me/c/2795094459/6
const CHANNEL_ID_SUMBER = -1002795094459; // ID dari channel/grup privat (diawali -100)
const MESSAGE_ID_SUMBER = 6;             // ID dari pesan spesifik yang akan disalin

/**
 * Memuat string sesi dari file lokal.
 * @returns {string} String sesi jika file ditemukan, atau string kosong jika tidak.
 */
function loadSession() {
  try {
    if (fs.existsSync(SESSION_FILE_PATH)) {
      const sessionString = fs.readFileSync(SESSION_FILE_PATH, "utf8");
      console.log("✅ Sesi ditemukan, mencoba memuat sesi...");
      return sessionString;
    }
  } catch (error) {
    console.error("❌ Gagal memuat sesi:", error);
  }
  console.log("ℹ️ Sesi tidak ditemukan, akan memulai otorisasi baru.");
  return "";
}

/**
 * Menyimpan string sesi ke file lokal.
 * @param {string} session - String sesi yang akan disimpan.
 */
function saveSession(session) {
  try {
    fs.writeFileSync(SESSION_FILE_PATH, session);
    console.log(`\n✅ Sesi berhasil disimpan ke file: ${SESSION_FILE_PATH}`);
    console.log("Anda tidak perlu login lagi pada proses berikutnya.");
  } catch (error) {
    console.error("❌ Gagal menyimpan sesi:", error);
  }
}

/**
 * Membaca daftar grup dari file `grup.txt`.
 * @returns {string[]} Array berisi daftar ID atau username grup.
 */
function dapatkanDaftarGrup() {
  try {
    const data = fs.readFileSync(GRUP_FILE, "utf8");
    return data.split(/\r?\n/).filter(line => line.trim() !== "");
  } catch (err) {
    console.error(`❌ Error: File '${GRUP_FILE}' tidak ditemukan.`);
    return [];
  }
}

/**
 * FUNGSI BARU: Mengambil objek pesan spesifik dari channel sumber.
 * @param {TelegramClient} client - Instance klien Telegram.
 * @returns {Promise<import('telegram/tl/types').Message|null>} Objek pesan jika berhasil, atau null jika gagal.
 */
async function ambilPesanSumber(client) {
    try {
        console.log(`\n🔄 Mengambil konten pesan sumber (ID: ${MESSAGE_ID_SUMBER}) dari channel (ID: ${CHANNEL_ID_SUMBER})...`);
        const messages = await client.getMessages(CHANNEL_ID_SUMBER, { ids: MESSAGE_ID_SUMBER });

        if (messages && messages.length > 0) {
            console.log("   ✅ Konten pesan berhasil diambil.");
            return messages[0]; // Mengembalikan objek pesan lengkap
        } else {
            console.error(`❌ Pesan dengan ID ${MESSAGE_ID_SUMBER} tidak ditemukan di channel.`);
            return null;
        }
    } catch (error) {
        console.error(`❌ Gagal mengambil pesan sumber: ${error.message}`);
        return null;
    }
}


/**
 * Fungsi utama untuk mengirim pesan ke semua grup dalam daftar.
 * @param {TelegramClient} client - Instance klien Telegram yang sudah terhubung.
 */
async function kirimPesanMassal(client) {
  // Ambil objek pesan sumber terlebih dahulu
  const pesanSumber = await ambilPesanSumber(client);

  if (!pesanSumber) {
      console.log("⚠️ Proses pengiriman dibatalkan karena gagal mengambil konten pesan sumber.");
      return;
  }

  console.log(`\n🚀 Memulai proses pengiriman pesan (copy-paste) pada ${new Date().toLocaleTimeString('id-ID')}`);
  const daftarGrup = dapatkanDaftarGrup();

  if (daftarGrup.length === 0) {
    console.log("⚠️ Tidak ada grup untuk dikirimi pesan. Periksa file 'grup.txt' Anda.");
    return;
  }

  for (const grup of daftarGrup) {
    try {
      let targetGrup;
      if (/^-?\d+$/.test(grup)) {
        targetGrup = parseInt(grup, 10);
      } else {
        targetGrup = grup;
      }

      // Mengirim pesan tanpa log per grup
      await client.sendMessage(targetGrup, pesanSumber);
      await new Promise(resolve => setTimeout(resolve, 5000));
    } catch (error) {
      // Tidak menampilkan log error per grup agar tidak menumpuk
    }
  }
  // Log penyelesaian dihapus agar terminal bersih
}

/**
 * Fungsi utama aplikasi yang dijalankan saat skrip dimulai.
 */
(async () => {
  const sessionString = loadSession();
  const client = new TelegramClient(new StringSession(sessionString), apiId, apiHash, {
    connectionRetries: 5,
  });

  await client.start({
    phoneNumber: async () => await input.text("Masukkan nomor telepon Anda: "),
    password: async () => await input.text("Masukkan kata sandi (2FA) Anda: "),
    phoneCode: async () => await input.text("Masukkan kode yang Anda terima: "),
    onError: (err) => console.log(err),
  });

  if (sessionString !== client.session.save()) {
    saveSession(client.session.save());
  } else {
    console.log("✅ Berhasil terhubung menggunakan sesi yang sudah ada.");
  }

  console.log("\n🚀 Mengirim pesan untuk pertama kalinya...");
  await kirimPesanMassal(client);

  cron.schedule('0 */3 * * *', () => {
    kirimPesanMassal(client);
  }, {
    scheduled: true,
    timezone: "Asia/Jakarta"
  });

  console.log("\n🕒 Penjadwalan berhasil! Pesan akan dikirim setiap 2 jam.");
  console.log(`Skrip akan tetap berjalan di latar belakang untuk menjaga jadwal.`);
})();
