require('dotenv').config();
const TelegramBot = require('node-telegram-bot-api');
const path = require('path');
const fs = require('fs');

// Import handler yang diperlukan
const apiHandler = require('./handlers/apiHandler');
const pricingHandler = require('./handlers/pricingHandler');
const bulkDropletHandler = require('./handlers/bulkDropletHandler');
const bulkWindowsHandler = require('./handlers/bulkWindowsHandler');
const dropletManagerHandler = require('./handlers/dropletManagerHandler');
const rcloneBackupHandler = require('./config/rcloneBackupHandler');
const autoDeleteHandler = require('./config/autoDeleteHandler');
const broadcastHandler = require('./handlers/broadcastHandler');
const { addUser, readUsersDb } = require('./config/users');
const googleSheets = require('./utils/googleSheets');

// --- PENGATURAN & VALIDASI ---
const token = process.env.TELEGRAM_BOT_TOKEN;
const AUTHORIZED_CHAT_ID = process.env.AUTHORIZED_CHAT_ID;

// Diagnostik Startup
console.log("==================== DIAGNOSTIK STARTUP ====================");
const cwd = process.cwd();
console.log(`[Diagnostik] Direktori kerja saat ini (CWD): ${cwd}`);
const envPath = path.join(cwd, '.env');
if (fs.existsSync(envPath)) {
    console.log(`[Diagnostik] File .env DITEMUKAN di: ${envPath}`);
} else {
    console.log(`[Diagnostik] KRITIS: File .env TIDAK DITEMUKAN di direktori kerja saat ini. Pastikan Anda menjalankan bot dari folder proyek yang benar.`);
}
console.log("============================================================");

console.log("[Startup Check] Memeriksa GOOGLE_SHEET_ID dari file .env...");
if (process.env.GOOGLE_SHEET_ID) {
    console.log(`[Startup Check] GOOGLE_SHEET_ID berhasil dimuat: ${process.env.GOOGLE_SHEET_ID}`);
} else {
    console.log("[Startup Check] PERINGATAN: GOOGLE_SHEET_ID tidak ditemukan atau kosong di file .env!");
}


const tripayCreds = ['TRIPAY_API_KEY', 'TRIPAY_PRIVATE_KEY', 'TRIPAY_MERCHANT_CODE'];
const missingCreds = tripayCreds.filter(cred => !process.env[cred]);

if (!token || !AUTHORIZED_CHAT_ID) {
    console.error("FATAL: Pastikan TELEGRAM_BOT_TOKEN dan AUTHORIZED_CHAT_ID sudah diatur di file .env!");
    process.exit(1);
}
if (missingCreds.length > 0) {
    console.error(`FATAL: Kredensial Tripay tidak lengkap. Harap atur: ${missingCreds.join(', ')} di file .env!`);
    process.exit(1);
}

const bot = new TelegramBot(token, { polling: true });
console.log('✅ Bot Telegram (Mode Publik & Admin) berhasil dijalankan!');

// --- MEMULAI PROSES LATAR BELAKANG ---
pricingHandler.initializePricingDb();
rcloneBackupHandler.startAutoBackup(10); // Mengaktifkan kembali rclone
autoDeleteHandler.startAutoDelete(bot, AUTHORIZED_CHAT_ID, 5);


// --- DEFINISI KEYBOARD BARU ---

const adminKeyboard = {
    reply_markup: {
        keyboard: [
            ['🪟 Create RDP', '🐧 Create VPS'],
            ['📦 Atur Produk', '🔑 Manajemen API'],
            ['🗑️ Hapus Droplet', '📋 List Droplet'],
            ['📢 Broadcast']
        ],
        resize_keyboard: true,
    }
};

const publicKeyboard = {
    reply_markup: {
        keyboard: [
            ['🪟 Buy RDP', '🐧 Buy VPS']
        ],
        resize_keyboard: true,
    }
};


// === PUSAT KONTROL EVENT (EVENT ROUTER) ===

bot.on('message', async (msg) => {
    const chatId = msg.chat.id;
    const text = msg.text;
    const isAdmin = String(chatId) === String(AUTHORIZED_CHAT_ID);
    const username = msg.from.username ? msg.from.username.replace(/_/g, '') : null;
    const firstName = msg.from.first_name || '';

    if (!text) return;

    // Tambahkan atau perbarui pengguna di database
    await addUser(chatId, username);

    // Pemeriksaan username untuk pengguna publik
    if (!isAdmin && (text === '🪟 Buy RDP' || text === '🐧 Buy VPS')) {
        if (!username) {
            bot.sendMessage(chatId, '⚠️ **Username Diperlukan!**\n\nUntuk melanjutkan pemesanan, Anda harus mengatur username di profil Telegram Anda.\n\nSilakan pergi ke **Pengaturan > Edit Profil > Username** untuk membuatnya, lalu coba lagi.', { parse_mode: 'HTML' });
            return;
        }
    }
    
    // Penanganan input teks untuk proses yang sedang berjalan
    if (await bulkDropletHandler.handleTextMessage(bot, msg)) return;
    if (await bulkWindowsHandler.handleTextMessage(bot, msg)) return;
    if (await broadcastHandler.handleTextMessage(bot, msg)) return;

    if (isAdmin) {
        if (await apiHandler.handleTextMessage(bot, msg)) return;
        if (await pricingHandler.handleTextMessage(bot, msg)) return;
    }

    // Penanganan perintah utama
    switch (text) {
        case '/start':
            const welcomeInfo = `Halo **${firstName}**, selamat datang di Toko JK Bot! 🤖\n\nKami adalah penyedia RDP & VPS berkualitas tinggi untuk segala kebutuhan Anda.\n\nBerikut adalah informasi akun Anda:
- **ID Tele:** \`${chatId}\`
- **Nama:** ${firstName}
- **Username:** ${username ? `@${username}` : '_(Tidak diatur)_'}`;
            
            const inlineKeyboard = {
                inline_keyboard: [
                    [{ text: '💬 Hubungi Admin', url: 'https://t.me/masjukii' }],
                    [{ text: '📢 Channel Kami', url: 'https://t.me/toko_jk' }, { text: '👥 Grup Komunitas', url: 'https://t.me/marketclouds' }]
                ]
            };

            await bot.sendMessage(chatId, welcomeInfo, { parse_mode: 'Markdown', reply_markup: inlineKeyboard });
            await bot.sendMessage(chatId, "Silakan pilih menu di bawah ini untuk memulai.", isAdmin ? adminKeyboard : publicKeyboard);
            break;

        case '🔑 Manajemen API':
            if (isAdmin) apiHandler.showApiMenu(bot, chatId);
            break;

        case '📦 Atur Produk':
            if (isAdmin) pricingHandler.showProductMenu(bot, chatId);
            break;
        
        case '🗑️ Hapus Droplet':
            if (isAdmin) dropletManagerHandler.showDropletList(bot, chatId, 1); // Mulai dari halaman 1
            break;

        case '📋 List Droplet':
            if (isAdmin) dropletManagerHandler.showFullDropletList(bot, chatId);
            break;

        case '📢 Broadcast':
            if (isAdmin) broadcastHandler.handleBroadcastCommand(bot, msg);
            break;
            
        case '💰 Cek Saldo':
            if (isAdmin) apiHandler.showAccountBalance(bot, chatId);
            break;
        
        case '⚙️ Cek Koneksi Sheet':
            if (isAdmin) googleSheets.testConnection(bot, chatId);
            break;

        // Rute untuk Publik
        case '🪟 Buy RDP':
            if (!isAdmin) bulkWindowsHandler.handleCreateWindows(bot, chatId, username, false);
            break;
        case '🐧 Buy VPS':
            if (!isAdmin) bulkDropletHandler.handleCreateDroplet(bot, chatId, username, false);
            break;

        // Rute untuk Admin
        case '🪟 Create RDP':
            if (isAdmin) bulkWindowsHandler.handleCreateWindows(bot, chatId, username, true);
            break;
        case '🐧 Create VPS':
            if (isAdmin) bulkDropletHandler.handleCreateDroplet(bot, chatId, username, true);
            break;
    }
});

bot.on('callback_query', async (callbackQuery) => {
    const chatId = callbackQuery.message.chat.id;
    const isAdmin = String(chatId) === String(AUTHORIZED_CHAT_ID);

    if (await bulkDropletHandler.handleCallbackQuery(bot, callbackQuery)) return;
    if (await bulkWindowsHandler.handleCallbackQuery(bot, callbackQuery)) return;
    if (await broadcastHandler.handleCallbackQuery(bot, callbackQuery)) return;

    if (isAdmin) {
        if (await apiHandler.handleCallbackQuery(bot, callbackQuery)) return;
        if (await pricingHandler.handleCallbackQuery(bot, callbackQuery)) return;
        if (await dropletManagerHandler.handleCallbackQuery(bot, callbackQuery)) return;
    }
});

bot.on('polling_error', (error) => {
    console.error(`Polling error: ${error.code} - ${error.message}`);
});

