const { readDb, writeDb } = require('./db');
const doApi = require('../utils/digitalocean');
const { updateRowStatusToDeleted } = require('../utils/googleSheets'); // Integrasi Google Sheets

/**
 * Mengubah timestamp UTC ke format Waktu Indonesia Barat (WIB).
 * @param {number} timestamp - Timestamp dalam detik.
 * @returns {string} Waktu dalam format HH:MM WIB.
 */
function toWibTime(timestamp) {
    const date = new Date(timestamp * 1000);
    // Menambahkan offset 7 jam untuk WIB
    date.setHours(date.getUTCHours() + 7);
    const hours = String(date.getUTCHours()).padStart(2, '0');
    const minutes = String(date.getUTCMinutes()).padStart(2, '0');
    return `${hours}.${minutes} WIB`;
}


/**
 * Memulai proses pengecekan dan penghapusan droplet otomatis secara berkala.
 * @param {object} bot - Instance bot Telegram.
 * @param {string} adminChatId - ID chat admin untuk notifikasi error.
 * @param {number} checkIntervalInMinutes - Interval pengecekan dalam menit.
 */
function startAutoDelete(bot, adminChatId, checkIntervalInMinutes) {
    // Penjelasan: Untuk mengubah interval menjadi 1 menit,
    // silakan ubah angka '5' menjadi '1' di file bot.js pada baris:
    // autoDeleteHandler.startAutoDelete(bot, AUTHORIZED_CHAT_ID, 5);
    console.log(`✅ Layanan auto-delete dimulai. Pengecekan setiap ${checkIntervalInMinutes} menit.`);
    
    // Logika baru untuk membersihkan console setiap 6 jam
    const sixHoursInMs = 6 * 60 * 60 * 1000;
    setInterval(() => {
        console.clear();
        console.log(`[Log Cleared] Console dibersihkan secara otomatis pada ${new Date().toLocaleString('id-ID', { timeZone: 'Asia/Jakarta' })}.`);
        console.log(`✅ Layanan auto-delete berjalan. Pengecekan setiap ${checkIntervalInMinutes} menit.`);
    }, sixHoursInMs);
    
    const runCheck = async () => {
        console.log(`[Auto-Delete] Menjalankan pengecekan pada ${new Date().toLocaleString('id-ID', { timeZone: 'Asia/Jakarta' })}...`);
        const db = await readDb();
        const now = Math.floor(Date.now() / 1000);
        const oneHourFromNow = now + 3600; // 3600 detik = 1 jam
        let dbChanged = false;
        
        for (const apiKeyName in db) {
            if (typeof db[apiKeyName] !== 'object' || !db[apiKeyName].token) continue;

            const apiData = db[apiKeyName];
            if (apiData.managed_droplets && apiData.token) {
                for (const dropletId in apiData.managed_droplets) {
                    const dropletInfo = apiData.managed_droplets[dropletId];
                    
                    // --- Logika Peringatan 1 Jam ---
                    if (dropletInfo.status === 'auto-delete-scheduled' && !dropletInfo.warning_sent && dropletInfo.delete_at_timestamp_utc < oneHourFromNow && dropletInfo.delete_at_timestamp_utc > now) {
                        const userChatId = dropletInfo.owner_chat_id;
                        if (userChatId) {
                            const warningMessage = `⚠️ <b>Peringatan Penghapusan RDP/VPS</b> ⚠️\n\nRDP/VPS Anda dengan IP <code>${dropletInfo.ip_address}</code> akan dihapus dalam waktu kurang dari 1 jam.\n\nHarap segera amankan data Anda.`;
                            bot.sendMessage(userChatId, warningMessage, { parse_mode: 'HTML' }).catch(e => console.error(`Gagal mengirim peringatan ke ${userChatId}:`, e.message));
                            
                            // Tandai bahwa peringatan sudah dikirim
                            db[apiKeyName].managed_droplets[dropletId].warning_sent = true;
                            dbChanged = true;
                            console.log(`[Auto-Delete] Peringatan 1 jam dikirim untuk droplet ${dropletInfo.name}.`);
                        }
                    }

                    // --- Logika Penghapusan ---
                    if (dropletInfo.status === 'auto-delete-scheduled' && dropletInfo.delete_at_timestamp_utc < now) {
                        console.log(`[Auto-Delete] Menghapus droplet ${dropletInfo.name} (${dropletId}) dari akun ${apiKeyName}.`);
                        
                        const result = await doApi.deleteDroplet(apiData.token, dropletId);
                        const userChatId = dropletInfo.owner_chat_id;
                        
                        if (result.success) {
                            console.log(`[Auto-Delete] Berhasil menghapus droplet ${dropletInfo.name}.`);
                            
                            // ---- INTEGRASI GOOGLE SHEETS ----
                            await updateRowStatusToDeleted(dropletInfo.ip_address);
                            // ---- AKHIR INTEGRASI ----

                            if (userChatId) {
                                const deletionTime = toWibTime(now);
                                const message = `🗑️ <b>RDP/VPS Telah Dihapus</b>\n\nIP: <code>${dropletInfo.ip_address}</code>\nPukul: ${deletionTime}`;
                                bot.sendMessage(userChatId, message, { parse_mode: 'HTML' }).catch(e => console.error(`Gagal mengirim notifikasi hapus ke ${userChatId}:`, e.message));
                            }
                            
                            delete db[apiKeyName].managed_droplets[dropletId];
                            dbChanged = true;
                        } else {
                            if (result.status === 404) {
                                console.log(`[Auto-Delete] Droplet ${dropletInfo.name} sudah tidak ada. Membersihkan dari database.`);
                                
                                // ---- INTEGRASI GOOGLE SHEETS ----
                                await updateRowStatusToDeleted(dropletInfo.ip_address);
                                // ---- AKHIR INTEGRASI ----
                                
                                delete db[apiKeyName].managed_droplets[dropletId];
                                dbChanged = true;
                            } else {
                                console.error(`[Auto-Delete] Gagal menghapus droplet ${dropletInfo.name}. Alasan: ${result.message}`);
                                const message = `❌ <b>Auto-Delete Gagal</b>\n\nGagal menghapus droplet <code>${dropletInfo.name}</code>.\n<b>Alasan:</b> ${result.message}`;
                                bot.sendMessage(adminChatId, message, { parse_mode: 'HTML' }).catch(e => console.error("Gagal mengirim notifikasi error ke admin:", e.message));
                            }
                        }
                    }
                }
            }
        }

        if (dbChanged) {
            await writeDb(db);
        }
    };

    runCheck();
    setInterval(runCheck, checkIntervalInMinutes * 60 * 1000);
}

module.exports = { startAutoDelete };

