const fs = require('fs');
const path = require('path');
const os = require('os');
const { exec } = require('child_process');
const crypto = require('crypto');

// Mendefinisikan path untuk semua file data yang akan di-backup
const dbPath = path.join(__dirname, '..', 'database.json');
const usersPath = path.join(__dirname, '..', 'users.json');
const hargaPath = path.join(__dirname, '..', 'database_harga.json'); // <-- FILE BARU DITAMBAHKAN
const stateFilePath = path.join(__dirname, '..', 'rclone_backup_state.json');

// Konfigurasi dari file .env
const remoteName = process.env.RCLONE_REMOTE_NAME || 'Backup';
const remotePath = process.env.RCLONE_REMOTE_PATH || 'DigitalOceanBot-Tester';

/**
 * Memastikan file konfigurasi rclone ada. Jika tidak, file akan dibuat.
 */
function ensureRcloneConfig() {
    const homeDir = os.homedir();
    const rcloneConfigDir = path.join(homeDir, '.config', 'rclone');
    const rcloneConfigFile = path.join(rcloneConfigDir, 'rclone.conf');

    if (fs.existsSync(rcloneConfigFile)) {
        // console.log('[Rclone Config] File rclone.conf sudah ada.'); // Dihilangkan agar log lebih bersih
        return;
    }

    console.log('[Rclone Config] File rclone.conf tidak ditemukan. Membuat konfigurasi baru...');
    const rcloneConfigContent = `[Backup]
type = drive
scope = drive
root_folder_id = 1qRyfhYB9TRTNSDinrkkTeyVoUUvZGG3K
token = {"access_token":"ya29.a0AS3H6Nyp3KVOZRPQeASwhwoCzFJntWEhi7E_DlBCSSt5QamS4wMqyl78OvhkH5YR1W2Z4IHR0RGgRC_AtzFR8e-7K556ASr9Yw-_qycKR2RXMEVEw33QdaSb1tvhQkgVgBWpiXkC-rbQYTPBPEUod6k6hqUyvckxHF3brA7qaCgYKARsSARQSFQHGX2Mio6iWZTAuPZwhh2K0zrTOdA0175","token_type":"Bearer","refresh_token":"1//0gnHr5v5lnUUZCgYIARAAGBASNwF-L9IrLEp43SYq3WESJExnG3ihKGuMuN7WDgEr9tFgBewPvTgJtaMlJ7Fg8gOEJqFJIXNXGDU","expiry":"2025-06-19T02:20:04.223867092+07:00"}
team_drive = 
`;

    try {
        fs.mkdirSync(rcloneConfigDir, { recursive: true });
        fs.writeFileSync(rcloneConfigFile, rcloneConfigContent);
        console.log('[Rclone Config] File rclone.conf berhasil dibuat di:', rcloneConfigFile);
    } catch (error) {
        console.error('[Rclone Config] Gagal membuat file rclone.conf:', error);
    }
}

/**
 * Menjalankan perintah shell sebagai Promise.
 */
function runCommand(command) {
    return new Promise((resolve, reject) => {
        exec(command, (error, stdout, stderr) => {
            if (error) {
                if (stderr.includes('doesn\'t exist')) {
                    return resolve('');
                }
                return reject(new Error(stderr || error.message));
            }
            resolve(stdout.trim());
        });
    });
}

/**
 * Membaca file state yang menyimpan hash terakhir.
 * @returns {object} Objek berisi hash file.
 */
function readStateFile() {
    try {
        if (fs.existsSync(stateFilePath)) {
            const data = fs.readFileSync(stateFilePath, 'utf8');
            return JSON.parse(data);
        }
    } catch (e) {
        console.error('[Rclone State] Gagal membaca file state, akan dibuat ulang.');
    }
    return {};
}

/**
 * Menulis hash baru ke file state.
 * @param {object} stateData Objek state yang akan disimpan.
 */
function writeStateFile(stateData) {
    fs.writeFileSync(stateFilePath, JSON.stringify(stateData, null, 2), 'utf8');
}


/**
 * Fungsi generik untuk menyinkronkan satu file dengan remote.
 * @param {string} localPath Path file lokal.
 */
async function syncFileWithRemote(localPath) {
    const fileName = path.basename(localPath);
    if (!fs.existsSync(localPath)) {
        return;
    }
    
    const localHash = crypto.createHash('md5').update(fs.readFileSync(localPath)).digest('hex');
    const state = readStateFile();
    const lastKnownHash = state[fileName];

    try {
        if (localHash !== lastKnownHash) {
            console.log(`[Rclone Backup] Perubahan lokal terdeteksi pada ${fileName}. Mengunggah...`);
            await runCommand(`rclone copy "${localPath}" ${remoteName}:${remotePath}`);
            state[fileName] = localHash;
            writeStateFile(state);
            // console.log(`[Rclone Backup] Unggah ${fileName} berhasil.`); // Dihilangkan agar log lebih bersih
        } else {
            const remoteHashOutput = await runCommand(`rclone md5sum ${remoteName}:${remotePath}/${fileName}`);
            const remoteHash = remoteHashOutput.split(' ')[0];

            if (remoteHash && remoteHash !== localHash) {
                console.log(`[Rclone Backup] Perubahan remote terdeteksi pada ${fileName}. Mengunduh...`);
                await runCommand(`rclone copy ${remoteName}:${remotePath}/${fileName} "${path.dirname(localPath)}"`);
                const newLocalHash = crypto.createHash('md5').update(fs.readFileSync(localPath)).digest('hex');
                state[fileName] = newLocalHash;
                writeStateFile(state);
                console.log(`[Rclone Backup] Unduh ${fileName} berhasil.`);
            } else if (!remoteHash) {
                 console.log(`[Rclone Backup] ${fileName} tidak ditemukan di remote. Mengunggah...`);
                 await runCommand(`rclone copy "${localPath}" ${remoteName}:${remotePath}`);
                 state[fileName] = localHash;
                 writeStateFile(state);
            }
        }
    } catch (error) {
        console.error(`[Rclone Backup] Terjadi kesalahan saat menyinkronkan ${fileName}:`, error.message);
    }
}


/**
 * Fungsi utama yang menjalankan proses backup untuk semua file yang ditentukan.
 */
async function performRcloneBackup() {
    console.log(`[Rclone Backup] Memulai pengecekan backup pada ${new Date().toLocaleString('id-ID')}`);

    // Sinkronkan semua file data penting
    await syncFileWithRemote(dbPath);
    await syncFileWithRemote(usersPath);
    await syncFileWithRemote(hargaPath); // <-- FILE BARU DITAMBAHKAN

    console.log('[Rclone Backup] Pengecekan backup selesai.');
}

/**
 * Memulai jadwal backup otomatis.
 * @param {number} intervalMinutes Interval dalam menit.
 */
function startAutoBackup(intervalMinutes = 10) {
    ensureRcloneConfig();
    
    console.log(`[Rclone Backup] Auto-backup via rclone aktif, berjalan setiap ${intervalMinutes} menit.`);
    
    // Jalankan pengecekan pertama kali saat bot dimulai
    performRcloneBackup();

    // Atur interval untuk pengecekan berikutnya
    setInterval(performRcloneBackup, intervalMinutes * 60 * 1000);
}

module.exports = { startAutoBackup };

