const { readDb, writeDb } = require('../config/db');
const doApi = require('../utils/digitalocean');
const tripay = require('../utils/tripay');
const { calculateTotalPrice } = require('../config/pricing');
const { ALLOWED_REGIONS, ALLOWED_SIZES } = require('../config/predefinedOptions');
const axios = require('axios');
const { readPricingDb } = require('./pricingHandler');
const googleSheets = require('../utils/googleSheets');
const { findUserByUsername } = require('../config/users');

const creationState = {};

function formatTimestamp(timestamp) {
    if (!timestamp) return 'N/A';
    const date = new Date(timestamp * 1000);
    const options = {
        timeZone: 'Asia/Jakarta',
        year: 'numeric', month: '2-digit', day: '2-digit',
        hour: '2-digit', minute: '2-digit', hour12: false
    };
    return new Intl.DateTimeFormat('sv-SE', options).format(date);
}


async function waitForDropletActive(token, dropletId) {
    const maxRetries = 24;
    const retryInterval = 10000;

    for (let i = 0; i < maxRetries; i++) {
        const dropletDetails = await doApi.getDropletDetails(token, dropletId);
        if (dropletDetails.success) {
            const droplet = dropletDetails.data.droplet;
            if (droplet.status === 'active') {
                const publicIp = droplet.networks.v4.find(net => net.type === 'public')?.ip_address;
                if (publicIp) {
                    return { success: true, ip: publicIp };
                }
            }
        }
        await new Promise(resolve => setTimeout(resolve, retryInterval));
    }
    return { success: false, message: 'Gagal mendapatkan IP setelah 4 menit, silahkan hubungi admin @masjukii' };
}

async function findAndConfirmApi(bot, chatId) {
    const state = creationState[chatId];
    const searchingMsg = await bot.sendMessage(chatId, 'Menghubungkan ke server, harap tunggu...');

    const db = await readDb();
    const allApiKeys = Object.keys(db);
    
    let keysToSearch;
    let apiType;

    if (state.daysToDelete <= 7) {
        keysToSearch = allApiKeys.filter(key => db[key].type === 'harian');
        apiType = 'Harian';
    } else {
        keysToSearch = allApiKeys.filter(key => db[key].type === 'bulanan');
        apiType = 'Bulanan';
    }

    if (keysToSearch.length === 0) {
        await bot.editMessageText(`❌ Server telah mencapai batas limit. Silakan hubungi admin @masjukii .`, { chat_id: chatId, message_id: searchingMsg.message_id });
        delete creationState[chatId];
        return;
    }
    
    const creationPlan = [];
    let remainingDropletsToPlan = state.count;

    const singleDropletCostUSD = (state.price_monthly_usd / 30) * state.daysToDelete;

    console.log(`\n--- Memulai Pencarian API Multi-Akun (Tipe: ${apiType}) ---`);
    console.log(`Pesanan: ${state.count} VPS, Durasi: ${state.daysToDelete} hari, Biaya per Droplet: $${singleDropletCostUSD.toFixed(2)}`);

    for (const keyName of keysToSearch) {
        if (remainingDropletsToPlan <= 0) break;

        console.log(`\n[Menganalisis Akun: ${keyName}]`);
        if (keyName === 'pending_payments') continue;

        const token = db[keyName].token;
        if (!token) continue;
        
        const [statusRes, balanceRes, dropletsRes, allSizesRes] = await Promise.all([
            doApi.getAccountStatus(token),
            doApi.getAccountBalance(token),
            doApi.getDroplets(token),
            doApi.getSizes(token)
        ]);

        if (!statusRes.success || !balanceRes.success || !dropletsRes.success || !allSizesRes.success) {
            console.log(` -> Gagal mengambil data awal, dilewati.`);
            continue;
        }

        const dropletLimit = statusRes.data.droplet_limit;
        const currentDropletsCount = dropletsRes.data.length;
        const availableSlots = dropletLimit - currentDropletsCount;
        console.log(` -> Slot Droplet: ${currentDropletsCount}/${dropletLimit}. Slot Tersedia: ${availableSlots}`);

        if (availableSlots > 0) {
            const accountBalance = parseFloat(balanceRes.data.account_balance);
            const availableCredits = accountBalance < 0 ? Math.abs(accountBalance) : 0;
            console.log(` -> Sisa Kredit Terdeteksi: $${availableCredits.toFixed(2)}`);
            
            let potentialUsage = 0;
            const managedDroplets = db[keyName].managed_droplets || {};
            const nowInSeconds = Math.floor(Date.now() / 1000);
            const allSizes = allSizesRes.data;

            for (const existingDroplet of dropletsRes.data) {
                const managedInfo = managedDroplets[existingDroplet.id];
                const sizeInfo = allSizes.find(s => s.slug === existingDroplet.size_slug);

                if (managedInfo && sizeInfo && managedInfo.delete_at_timestamp_utc) {
                    const remainingSeconds = managedInfo.delete_at_timestamp_utc - nowInSeconds;
                    if (remainingSeconds > 0) {
                        const remainingDays = remainingSeconds / (24 * 3600);
                        const dailyPrice = sizeInfo.price_monthly / 30;
                        potentialUsage += dailyPrice * remainingDays;
                    }
                }
            }
            console.log(` -> Biaya Terpesan (Droplet Aktif): $${potentialUsage.toFixed(2)}`);
            
            const remainingBudget = availableCredits - potentialUsage;
            console.log(` -> Sisa Budget Sebenarnya: $${remainingBudget.toFixed(2)}`);
            
            const maxCanAfford = singleDropletCostUSD > 0 ? Math.floor(remainingBudget / singleDropletCostUSD) : availableSlots;
            console.log(` -> Maksimal bisa dibuat (dana): ${maxCanAfford}`);

            const numToCreateOnThisApi = Math.min(remainingDropletsToPlan, availableSlots, maxCanAfford);

            if (numToCreateOnThisApi > 0) {
                console.log(` -> KEPUTUSAN: Membuat ${numToCreateOnThisApi} droplet di akun ini.`);
                creationPlan.push({
                    apiKeyName: keyName,
                    token: token,
                    count: numToCreateOnThisApi
                });
                remainingDropletsToPlan -= numToCreateOnThisApi;
            } else {
                console.log(` -> KEPUTUSAN: Tidak membuat droplet (slot/dana tidak cukup).`);
            }
        }
    }

    console.log('--- Pencarian Selesai ---');
    await bot.deleteMessage(chatId, searchingMsg.message_id);

    if (remainingDropletsToPlan > 0) {
        console.log(`Gagal merencanakan ${remainingDropletsToPlan} droplet.`);
        await bot.sendMessage(chatId, `❌ Maaf,anda hanya bisa membuat VPS sebanyak ${state.count - remainingDropletsToPlan} dari ${state.count} VPS yang Anda minta. Silakan hubungi admin @masjukii.`);
        delete creationState[chatId];
    } else {
        console.log('Rencana pembuatan berhasil dibuat:', creationPlan);
        state.creationPlan = creationPlan;
        state.step = 'confirm';
        await showConfirmation(bot, chatId);
    }
}


// === FUNGSI-FUNGSI WIZARD (PEMANDU PEMBUATAN) ===

async function askForRegion(bot, chatId) {
    const keyboard = ALLOWED_REGIONS.map(r => ([{ text: `📍 ${r.name}`, callback_data: `bulk_linux_region_${r.slug}` }]));
    keyboard.push([{ text: '⛔ Batalkan', callback_data: 'bulk_linux_cancel' }]);
    await bot.sendMessage(chatId, '<b>Pilih region</b>', { parse_mode: 'HTML', reply_markup: { inline_keyboard: keyboard } });
}

async function askForSize(bot, chatId, isAdmin) {
    const pricingData = await readPricingDb();
    let availableSizes = 0;
    
    const keyboard = ALLOWED_SIZES.map(s => {
        const productInfo = pricingData[s.slug];
        if (!productInfo) return null;

        if (isAdmin || productInfo.enabled) {
            availableSizes++;
            const displayText = `${s.memory / 1024}GB/${s.vcpus}vCPU / ${s.disk} GB Disk`;
            let status = '';
            if (isAdmin) {
                status = productInfo.enabled ? '(✅ Aktif)' : '(❌ Nonaktif)';
            }
            return [{ text: `${displayText} ${status}`, callback_data: `bulk_linux_size_${s.slug}` }];
        }
        return null;
    }).filter(row => row !== null);

    if (availableSizes === 0) {
        await bot.sendMessage(chatId, 'ℹ️ Maaf, saat ini tidak ada produk VPS yang tersedia.');
        delete creationState[chatId];
        return;
    }

    keyboard.push([{ text: '⛔ Batalkan', callback_data: 'bulk_linux_cancel' }]);
    await bot.sendMessage(chatId, '<b>Pilih ukuran VPS</b>', { parse_mode: 'HTML', reply_markup: { inline_keyboard: keyboard } });
}

async function askForImage(bot, chatId) {
    const keyboard = [
        [
            { text: 'Ubuntu 24.04', callback_data: 'bulk_linux_image_ubuntu-24-04-x64' },
            { text: 'Ubuntu 22.04', callback_data: 'bulk_linux_image_ubuntu-22-04-x64' }
        ],
        [
            { text: 'Debian 12', callback_data: 'bulk_linux_image_debian-12-x64' }
        ],
        
        [{ text: '⛔ Batalkan', callback_data: 'bulk_linux_cancel' }]
    ];
    await bot.sendMessage(chatId, '<b>Pilih OS</b>', { parse_mode: 'HTML', reply_markup: { inline_keyboard: keyboard } });
}

async function showConfirmation(bot, chatId) {
    const state = creationState[chatId];
    const durationText = `${state.daysToDelete} hari`;

    let text = `<b>📝 Konfirmasi Pesanan VPS</b>\n\n`;
    text += `Anda akan membuat <b>${state.count} VPS</b> dengan detail:\n`;
    text += `▪️ Region: ${state.regionName}\n`;
    text += `▪️ Ukuran: ${state.sizeDescription}\n`;
    text += `▪️ OS: ${state.imageName}\n`;
    text += `▪️ Password: ${state.password}\n`;
    text += `▪️ Durasi: ${durationText}\n\n`;
    
    // Tambahkan info target pengguna jika ada
    if (state.targetUsername) {
        text += `👤 Target Pengguna: @${state.targetUsername}\n\n`;
    }

    let keyboard;

    if (state.isAdmin) {
        text += `Lanjutkan pembuatan?`;
        keyboard = [[{ text: '✅ Ya, Buat Sekarang', callback_data: 'bulk_linux_admin_create' }, { text: '⛔ Batalkan', callback_data: 'bulk_linux_cancel' }]];
    } else {
        const finalTotalPrice = calculateTotalPrice(state.price_min_idr, state.price_max_idr, state.daysToDelete) * state.count;
        state.finalTotalPrice = finalTotalPrice;
        text += `💰 <b>Total Biaya: Rp ${finalTotalPrice.toLocaleString('id-ID')}</b>\n\n`;
        text += `Lanjutkan ke pembayaran?`;
        keyboard = [[{ text: '✅ Ya, Lanjut Bayar', callback_data: 'bulk_linux_confirm_yes' }, { text: '⛔ Batalkan', callback_data: 'bulk_linux_cancel' }]];
    }

    await bot.sendMessage(chatId, text, { parse_mode: 'HTML', reply_markup: { inline_keyboard: keyboard } });
}

async function startPaymentPolling(bot, chatId, reference, qrMessageId) {
    const state = creationState[chatId];
    if (!state) return;

    const maxDuration = 10 * 60 * 1000;
    const intervalTime = 10000;
    const startTime = Date.now();

    const poller = setInterval(async () => {
        if (!creationState[chatId]) {
            clearInterval(poller);
            return;
        }
        
        if (Date.now() - startTime > maxDuration) {
            clearInterval(poller);
            delete creationState[chatId];
            await bot.deleteMessage(chatId, qrMessageId).catch(() => {});
            await bot.sendMessage(chatId, '⚠️ Waktu pembayaran telah habis. Silakan mulai lagi dari awal.');
            return;
        }

        const statusResult = await tripay.checkTransactionStatus(reference);
        if (statusResult.success && statusResult.data.status === 'PAID') {
            clearInterval(poller);
            await bot.deleteMessage(chatId, qrMessageId).catch(() => {});
            await startCreationProcess(bot, chatId);
        } else if (statusResult.success && statusResult.data.status === 'EXPIRED') {
            clearInterval(poller);
            delete creationState[chatId];
            await bot.deleteMessage(chatId, qrMessageId).catch(() => {});
            await bot.sendMessage(chatId, '⚠️ Transaksi ini telah kedaluwarsa. Silakan mulai lagi dari awal.');
        }
    }, intervalTime);

    state.paymentPoller = poller;
}

async function askForPaymentMethod(bot, chatId) {
    const state = creationState[chatId];
    const totalPrice = state.finalTotalPrice;
    const durationText = `${state.daysToDelete} hari`;
    const customer = { name: 'User ' + chatId, email: `user${chatId}@tokojk.com`, phone: '082143926830' };
    const orderItems = [{
        sku: state.size,
        name: `VPS ${state.sizeDescription} (${durationText})`,
        price: totalPrice / state.count,
        quantity: state.count
    }];

    const paymentMsg = await bot.sendMessage(chatId, '⏳ Sedang membuat kode QR pembayaran...');

    const trx = await tripay.requestTransaction(totalPrice, 'QRIS', customer, orderItems);
    
    await bot.deleteMessage(chatId, paymentMsg.message_id);

    if (!trx.success || !trx.data.qr_url) {
        await bot.sendMessage(chatId, `❌ Gagal membuat pembayaran: ${trx.message || 'Tidak ada URL QR.'}`);
        delete creationState[chatId];
        return;
    }

    state.tripayReference = trx.data.reference;
    state.step = 'awaiting_payment';

    let caption = `<b>💳 Silakan Scan QRIS untuk Pembayaran</b>\n\n`;
    caption += `<b>Total:</b> Rp ${trx.data.amount.toLocaleString('id-ID')}\n\n`;
    caption += `Qris akan di hapus otomatis dalam 10 menit.`;

    const keyboard = [
        [{ text: '⛔ Batalkan', callback_data: 'bulk_linux_cancel' }]
    ];

    try {
        const imageResponse = await axios.get(trx.data.qr_url, { responseType: 'arraybuffer' });
        const imageBuffer = Buffer.from(imageResponse.data, 'binary');

        const qrMessage = await bot.sendPhoto(chatId, imageBuffer, {
            caption: caption,
            parse_mode: 'HTML',
            reply_markup: { inline_keyboard: keyboard }
        });
        
        await startPaymentPolling(bot, chatId, trx.data.reference, qrMessage.message_id);

    } catch (e) {
        console.error("Gagal mengirim foto QRIS:", e.message);
        await bot.sendMessage(chatId, `❌ Terjadi kesalahan saat menampilkan QRIS. Silakan coba lagi atau hubungi admin.`);
        delete creationState[chatId];
    }
}

async function startCreationProcess(bot, chatId) {
    const state = { ...creationState[chatId] };
    delete creationState[chatId];
    
    // Tentukan ID chat tujuan akhir
    const finalTargetChatId = state.targetChatId || chatId;
    
    if (state.isAdmin) {
        const adminMsg = `🚀 Memulai proses pembuatan <b>${state.count}</b> VPS...` +
                         (state.targetUsername ? `\nTarget: @${state.targetUsername}` : '');
        await bot.sendMessage(chatId, adminMsg, { parse_mode: 'HTML' });
    } else {
        await bot.sendMessage(chatId, `Pembayaran berhasil! Memulai proses pembuatan <b>${state.count}</b> VPS...`, { parse_mode: 'HTML' });
    }

    const db = await readDb();
    const allFinalResults = [];
    let nameCounter = 1;

    for (const plan of state.creationPlan) {
        const { apiKeyName, token, count } = plan;
        const creationPromises = Array.from({ length: count }, () => {
            // Gunakan username target sebagai nama dasar jika ada
            const baseName = state.targetUsername || state.baseName;
            const name = `${baseName}-${nameCounter++}`;
            const userData = `#!/bin/bash
set -ex
echo "root:${state.password}" | chpasswd
`;
            const options = { name, region: state.region, size: state.size, image: state.image, tags: ['linux-bot-created'], user_data: userData };
            return doApi.createDroplet(token, options);
        });

        const creationResults = await Promise.all(creationPromises);
        
        const waitingPromises = creationResults.map(async (res, i) => {
            const baseName = state.targetUsername || state.baseName;
            const name = `${baseName}-${(nameCounter - count) + i}`;
            if (res.success) {
                const dropletId = res.data.droplet.id;
                const ipResult = await waitForDropletActive(token, dropletId);
                return { apiKeyName, name, success: ipResult.success, ip: ipResult.ip, message: ipResult.message, id: dropletId };
            } else {
                return { apiKeyName, name, success: false, message: res.message };
            }
        });

        const finalResultsForThisApi = await Promise.all(waitingPromises);
        allFinalResults.push(...finalResultsForThisApi);
    }
    
    let successfulCreations = [];
    let failedCreations = [];

    for (const res of allFinalResults) {
        if (res.success) {
            successfulCreations.push(res);
            const { apiKeyName, id, name, ip } = res;
            if (!db[apiKeyName].managed_droplets) db[apiKeyName].managed_droplets = {};
            
            const info = { 
                name, 
                os: state.imageName,
                creation_timestamp_utc: Math.floor(Date.now() / 1000), 
                status: "auto-delete-scheduled",
                owner_chat_id: finalTargetChatId, // Gunakan ID chat target
                ip_address: ip,
            };
            info.delete_after_days = state.daysToDelete;
            const thirtyMinutesInSeconds = 30 * 60;
            info.delete_at_timestamp_utc = info.creation_timestamp_utc + (state.daysToDelete * 24 * 60 * 60) + thirtyMinutesInSeconds;
            db[apiKeyName].managed_droplets[id] = info;

            try {
                const expiredDate = formatTimestamp(info.delete_at_timestamp_utc);
                const rowData = [
                    new Date().toLocaleString('id-ID', { timeZone: 'Asia/Jakarta' }),
                    'VPS',
                    info.name,
                    state.sizeDescription,
                    state.regionName,
                    info.ip_address,
                    info.os,
                    state.password,
                    expiredDate,
                    apiKeyName,
                    String(finalTargetChatId), // Gunakan ID chat target untuk sheet
                    'Aktif'
                ];
                await googleSheets.appendRow(rowData);
            } catch (sheetError) {
                console.error(`[Google Sheets] Gagal mencatat VPS ke sheet: ${sheetError.message}`);
            }

        } else {
            failedCreations.push(res);
        }
    }
    
    if (successfulCreations.length > 0) await writeDb(db);
    
    let reportText = '';
    if (successfulCreations.length > 0) {
        reportText += `✅ Berhasil Membeli ${successfulCreations.length} VPS:\n\n`;
        successfulCreations.forEach(d => {
            reportText += `OS: ${state.imageName}\n`;
            reportText += `IP: <code>${d.ip}</code>\n`;
            reportText += `User: <code>root</code>\n`;
            reportText += `Pass: <code>${state.password}</code>\n\n`;
        });
    }
    if (failedCreations.length > 0) {
        reportText += `❌ Gagal (${failedCreations.length}):\n`;
        failedCreations.forEach(f => {
            reportText += `- ${f.name} (Alasan: ${f.reason})\n`;
        });
    }
    reportText += 'INFO: Instalasi VPS akan berjalan sekitar 5-10 menit.\n\n';
    reportText += 'NOTED: Jangan pernah menyimpan data penting anda di VPS, untuk menghindari vps sewaktu" bisa mati tanpa pemberitahuan.';
    // Kirim laporan ke pengguna akhir
    await bot.sendMessage(finalTargetChatId, reportText, { parse_mode: 'HTML' });

    // Jika admin membuat untuk orang lain, kirim juga notifikasi singkat ke admin
    if (state.isAdmin && state.targetUsername) {
        await bot.sendMessage(chatId, `✅ Laporan pembuatan VPS telah dikirim ke @${state.targetUsername}.`);
    }
}

async function handleCreateDroplet(bot, chatId, username, isAdmin) {
    delete creationState[chatId];
    
    creationState[chatId] = { 
        step: isAdmin ? 'ask_target_user' : 'ask_count', 
        handler: 'linux', 
        username: username || `user${chatId}`,
        isAdmin: isAdmin
    };
    
    if (isAdmin) {
        const keyboard = [[
            { text: '👤 Untuk Pengguna Lain', callback_data: 'bulk_linux_target_other' },
            { text: '👨‍💻 Untuk Saya Sendiri', callback_data: 'bulk_linux_target_self' }
        ]];
        await bot.sendMessage(chatId, '🐧 **Create VPS (Admin)**\n\nAnda ingin membuat VPS ini untuk siapa?', {
            parse_mode: 'HTML',
            reply_markup: { inline_keyboard: keyboard }
        });
    } else {
        await bot.sendMessage(chatId, `<b>🐧 Buy VPS</b>\n\nBerapa banyak VPS yang ingin Anda buat?\nKetik angka, misal: 2`, { parse_mode: 'HTML' });
    }
}

async function handleTextMessage(bot, msg) {
    const chatId = msg.chat.id;
    if (!creationState[chatId] || creationState[chatId].handler !== 'linux') return false;

    const state = creationState[chatId];
    const text = msg.text;

    try {
        if (state.step === 'ask_target_username') {
            const targetUsername = text.replace('@', '');
            const foundUser = await findUserByUsername(targetUsername);

            if (!foundUser) {
                await bot.sendMessage(chatId, `⚠️ Pengguna dengan username @${targetUsername} tidak ditemukan di database. Pastikan pengguna tersebut pernah berinteraksi dengan bot.`);
                return true;
            }

            state.targetUsername = foundUser.username;
            state.targetChatId = foundUser.id;
            state.step = 'ask_count';
            await bot.sendMessage(chatId, `✅ Pengguna @${targetUsername} ditemukan. Berapa banyak VPS yang ingin Anda buat untuknya?`);
        
        } else if (state.step === 'ask_count') {
            const count = parseInt(text, 10);
            if (isNaN(count) || count < 1 || count > 10) {
                await bot.sendMessage(chatId, '⚠️ Jumlah tidak valid. Harap masukkan angka antara 1 dan 10.');
                return true;
            }
            state.count = count;
            state.baseName = state.username;
            state.step = 'select_region';
            await askForRegion(bot, chatId);
        } else if (state.step === 'ask_password') {
            state.password = text;
            state.step = 'ask_days';
            await bot.sendMessage(chatId, `<b>Masukkan durasi VPS</b> \nJangka waktu 1-7 hari.`, { parse_mode: 'HTML' });
        } else if (state.step === 'ask_days') {
            const days = parseInt(text, 10);
            if (isNaN(days)) {
                await bot.sendMessage(chatId, '⚠️ Input tidak valid. Harap masukkan angka.');
                return true;
            }
            if (days === 0) {
                delete creationState[chatId];
                await bot.sendMessage(chatId, 'Pembuatan VPS dibatalkan.');
                return true;
            }
            if (days < 1 || days > 7) {
                await bot.sendMessage(chatId, '⚠️ Durasi harus antara 1 hingga 7 hari. Hubungi admin @masjukii untuk durasi lebih lama..');
                return true;
            }
            state.daysToDelete = days;
            await findAndConfirmApi(bot, chatId);
        }
    } catch (e) { console.error(e); delete creationState[chatId]; }
    return true;
}

async function handleCallbackQuery(bot, cbq) {
    const data = cbq.data;
    if (!data.startsWith('bulk_linux_')) return false;

    const msg = cbq.message;
    const chatId = msg.chat.id;
    
    if (!msg.photo) {
        await bot.deleteMessage(chatId, msg.message_id).catch(() => {});
    }
    
    if (data === 'bulk_linux_cancel') {
        const state = creationState[chatId];
        if (state && state.paymentPoller) {
            clearInterval(state.paymentPoller);
        }
        if (msg.photo) {
            await bot.deleteMessage(chatId, msg.message_id).catch(() => {});
        }
        delete creationState[chatId];
        await bot.sendMessage(chatId, 'Pembuatan VPS dibatalkan.');
        return true;
    }

    const state = creationState[chatId];
    if (!state) return true;

    try {
        const step = state.step;
        if (step === 'ask_target_user' && data === 'bulk_linux_target_other') {
            state.step = 'ask_target_username';
            await bot.sendMessage(chatId, 'Silakan masukkan username Telegram pengguna (diawali @ atau tidak).');
        } else if (step === 'ask_target_user' && data === 'bulk_linux_target_self') {
            state.step = 'ask_count';
            await bot.sendMessage(chatId, `Berapa banyak VPS yang ingin Anda buat untuk diri sendiri?`);
        } else if (step === 'select_region' && data.startsWith('bulk_linux_region_')) {
            const regionSlug = data.replace('bulk_linux_region_', '');
            const regionDetails = ALLOWED_REGIONS.find(r => r.slug === regionSlug);
            state.region = regionSlug;
            state.regionName = regionDetails ? regionDetails.name : regionSlug;
            state.step = 'select_size';
            await askForSize(bot, chatId, state.isAdmin);
        } else if (step === 'select_size' && data.startsWith('bulk_linux_size_')) {
            const slug = data.replace('bulk_linux_size_', '');
            const sizeDetails = ALLOWED_SIZES.find(s => s.slug === slug);

            const pricingData = await readPricingDb();
            const priceInfo = pricingData[slug];
            if (!priceInfo) {
                 await bot.sendMessage(chatId, 'Error: Harga untuk ukuran ini tidak ditemukan.');
                 return true;
            }

            state.size = slug;
            state.sizeDescription = `${sizeDetails.memory / 1024}GB/${sizeDetails.vcpus}vCPU / ${sizeDetails.disk}GB Disk`;
            state.price_monthly_usd = sizeDetails.price_monthly_usd;
            state.price_min_idr = priceInfo.price_min_idr;
            state.price_max_idr = priceInfo.price_max_idr;

            state.step = 'select_image';
            await askForImage(bot, chatId);
        } else if (step === 'select_image' && data.startsWith('bulk_linux_image_')) {
            const imageSlug = data.replace('bulk_linux_image_', '');
            const button = cbq.message.reply_markup.inline_keyboard.flat().find(b => b.callback_data === data);
            state.image = imageSlug;
            state.imageName = button ? button.text : imageSlug;
            state.step = 'ask_password';
            await bot.sendMessage(chatId, `<b>Masukkan password</b>`, { parse_mode: 'HTML'});
        } else if (step === 'confirm' && data === 'bulk_linux_confirm_yes') {
            state.step = 'payment';
            await askForPaymentMethod(bot, chatId);
        } else if (step === 'confirm' && data === 'bulk_linux_admin_create') {
            if (state.isAdmin) {
                await startCreationProcess(bot, chatId);
            }
        }
    } catch (e) { console.error(e); delete creationState[chatId]; }
    return true;
}

module.exports = {
    handleCreateDroplet,
    handleTextMessage,
    handleCallbackQuery,
    startCreationProcess,
    creationState
};

