const { readDb, writeDb } = require('../config/db');
const doApi = require('../utils/digitalocean');
const googleSheets = require('../utils/googleSheets');

// State untuk menyimpan data sementara untuk paginasi
const listState = {};

/**
 * Mengonversi Unix timestamp (detik) ke format tanggal yang mudah dibaca (WIB).
 * @param {number} timestamp - Timestamp dalam detik.
 * @returns {string} Tanggal dalam format YYYY-MM-DD HH:mm.
 */
function formatTimestamp(timestamp) {
    if (!timestamp) return 'N/A';
    const date = new Date(timestamp * 1000);
    const options = {
        timeZone: 'Asia/Jakarta',
        year: 'numeric', month: '2-digit', day: '2-digit',
        hour: '2-digit', minute: '2-digit', hour12: false
    };
    return new Intl.DateTimeFormat('sv-SE', options).format(date);
}

/**
 * Fungsi baru untuk menampilkan daftar droplet per halaman API.
 * @param {object} bot - Instance bot Telegram.
 * @param {string|number} chatId - ID chat admin.
 * @param {number} apiIndex - Indeks dari akun API yang akan ditampilkan.
 * @param {number} messageId - ID pesan untuk diedit.
 */
async function displayApiListPage(bot, chatId, apiIndex = 0, messageId = null) {
    const db = await readDb();
    
    // Mengambil semua nama akun API yang memiliki droplet
    const apiKeys = Object.keys(db).filter(key => 
        db[key] && db[key].managed_droplets && Object.keys(db[key].managed_droplets).length > 0
    );

    if (apiKeys.length === 0) {
        await bot.editMessageText('ℹ️ Tidak ada RDP/VPS aktif yang dikelola oleh bot saat ini.', {
            chat_id: chatId,
            message_id: messageId
        });
        return;
    }

    // Memastikan indeks halaman valid
    apiIndex = Math.max(0, Math.min(apiIndex, apiKeys.length - 1));
    const currentApiKey = apiKeys[apiIndex];
    
    const droplets = [];
    for (const dropletId in db[currentApiKey].managed_droplets) {
        droplets.push({ id: dropletId, ...db[currentApiKey].managed_droplets[dropletId] });
    }
    droplets.sort((a, b) => b.creation_timestamp_utc - a.creation_timestamp_utc);

    let messageText = `<b>📋 Daftar RDP/VPS untuk Akun API:</b>\n`;
    messageText += `<b>${currentApiKey}</b> (Akun ${apiIndex + 1}/${apiKeys.length})\n\n`;
    
    droplets.forEach((droplet, index) => {
        const expiredDate = formatTimestamp(droplet.delete_at_timestamp_utc);
        messageText += `<b>${index + 1}. ${droplet.name || 'Tanpa Nama'}</b>\n`;
        messageText += ` ├─ IP: <code>${droplet.ip_address || 'N/A'}</code>\n`;
        messageText += ` └─ Kedaluwarsa: ${expiredDate} WIB\n\n`;
    });

    const MAX_LENGTH = 4096;
    if (messageText.length > MAX_LENGTH) {
        messageText = messageText.substring(0, 4050) + "\n\n...(Daftar terpotong karena terlalu panjang)...";
    }

    const navigationRow = [];
    if (apiIndex > 0) {
        navigationRow.push({ text: '⬅️ API Seb.', callback_data: `droplet_list:page:${apiIndex - 1}` });
    }
    if (apiIndex < apiKeys.length - 1) {
        navigationRow.push({ text: 'API Sel. ➡️', callback_data: `droplet_list:page:${apiIndex + 1}` });
    }
    
    await bot.editMessageText(messageText, {
        chat_id: chatId,
        message_id: messageId,
        parse_mode: 'HTML',
        reply_markup: { inline_keyboard: [navigationRow] }
    }).catch(e => {
        if (!e.message.includes('message is not modified')) console.error("Error paginasi API:", e);
    });
}

/**
 * Titik masuk utama untuk menampilkan daftar lengkap RDP/VPS, sekarang menggunakan sistem halaman per API.
 */
async function showFullDropletList(bot, chatId) {
    const loadingMsg = await bot.sendMessage(chatId, '⏳ Mengelompokkan daftar RDP/VPS berdasarkan Akun API...');
    await displayApiListPage(bot, chatId, 0, loadingMsg.message_id);
}

/**
 * Menampilkan daftar semua droplet yang dikelola bot dengan paginasi (untuk HAPUS).
 */
async function showDropletList(bot, chatId, page = 1, messageId = null) {
    if (page === 1 && !messageId) {
        const sentMessage = await bot.sendMessage(chatId, '⏳ Mengambil daftar droplet, harap tunggu...');
        messageId = sentMessage.message_id;
    }

    const db = await readDb();
    
    const allDroplets = [];
    for (const apiKeyName in db) {
        const apiData = db[apiKeyName];
        if (apiData && apiData.managed_droplets) {
            for (const dropletId in apiData.managed_droplets) {
                const dropletInfo = apiData.managed_droplets[dropletId];
                allDroplets.push({ id: dropletId, apiKeyName, ...dropletInfo });
            }
        }
    }

    if (allDroplets.length === 0) {
        await bot.editMessageText('ℹ️ Tidak ada RDP/VPS aktif yang dikelola oleh bot saat ini.', {
            chat_id: chatId,
            message_id: messageId
        });
        return;
    }

    allDroplets.sort((a, b) => b.creation_timestamp_utc - a.creation_timestamp_utc);
    
    const ITEMS_PER_PAGE = 8;
    const totalPages = Math.ceil(allDroplets.length / ITEMS_PER_PAGE);
    page = Math.max(1, Math.min(page, totalPages)); 

    const startIndex = (page - 1) * ITEMS_PER_PAGE;
    const dropletsOnPage = allDroplets.slice(startIndex, startIndex + ITEMS_PER_PAGE);

    let messageText = `<b>🗑️ Hapus RDP/VPS (Hal ${page}/${totalPages})</b>\n\nPilih salah satu untuk dihapus:`;
    const keyboard = [];

    dropletsOnPage.forEach(droplet => {
        const buttonText = `❌ ${droplet.name || droplet.id} (${droplet.ip_address || 'IP N/A'})`;
        keyboard.push([{ text: buttonText, callback_data: `droplet_delete:confirm:${droplet.apiKeyName}:${droplet.id}` }]);
    });
    
    const navigationRow = [];
    if (page > 1) {
        navigationRow.push({ text: '⬅️ Sebelumnya', callback_data: `droplet_delete:page:${page - 1}` });
    }
    if (page < totalPages) {
        navigationRow.push({ text: 'Berikutnya ➡️', callback_data: `droplet_delete:page:${page + 1}` });
    }
    if (navigationRow.length > 0) {
        keyboard.push(navigationRow);
    }
    
    await bot.editMessageText(messageText, {
        chat_id: chatId,
        message_id: messageId,
        parse_mode: 'HTML',
        reply_markup: { inline_keyboard: keyboard }
    }).catch(e => {
        if (!e.message.includes('message is not modified')) console.error("Error paginasi:", e);
    });
}

/**
 * Menangani semua callback query dari handler ini.
 */
async function handleCallbackQuery(bot, cbq) {
    const data = cbq.data;
    const chatId = cbq.message.chat.id;

    // Handler untuk navigasi daftar droplet LENGKAP
    if (data.startsWith('droplet_list:')) {
        const parts = data.split(':');
        const action = parts[1];

        if (action === 'page') {
            const apiIndex = parseInt(parts[2], 10);
            await displayApiListPage(bot, chatId, apiIndex, cbq.message.message_id);
        }
        return true;
    }

    // Handler untuk daftar HAPUS droplet
    if (data.startsWith('droplet_delete:')) {
        const parts = data.split(':');
        const action = parts[1];

        if (action === 'page') {
            const page = parseInt(parts[2], 10);
            await showDropletList(bot, chatId, page, cbq.message.message_id);
            return true;
        }
        
        const apiKeyName = parts[2];
        const dropletId = parts[3];

        await bot.answerCallbackQuery(cbq.id);

        if (action === 'confirm') {
            const db = await readDb();
            const dropletInfo = db[apiKeyName]?.managed_droplets?.[dropletId];

            if (!dropletInfo) {
                await bot.sendMessage(chatId, '❌ Gagal menemukan informasi droplet. Mungkin sudah dihapus.');
                return true;
            }

            const text = `⚠️ **Anda yakin ingin menghapus RDP/VPS ini?**\n\n` +
                         `<b>Nama:</b> ${dropletInfo.name}\n` +
                         `<b>IP:</b> ${dropletInfo.ip_address}\n\n` +
                         `Tindakan ini tidak dapat diurungkan.`;
            
            await bot.editMessageText(text, {
                chat_id: chatId,
                message_id: cbq.message.message_id,
                parse_mode: 'HTML',
                reply_markup: {
                    inline_keyboard: [
                        [{ text: '✅ Ya, Hapus Sekarang', callback_data: `droplet_delete:execute:${apiKeyName}:${dropletId}` }],
                        [{ text: '⛔ Batalkan', callback_data: `droplet_delete:page:1` }]
                    ]
                }
            });

        } else if (action === 'execute') {
            await bot.editMessageText(`⏳ Menghapus droplet...`, { chat_id: chatId, message_id: cbq.message.message_id, reply_markup: null });
            const db = await readDb();
            const token = db[apiKeyName]?.token;
            const dropletInfo = db[apiKeyName]?.managed_droplets?.[dropletId];
            const dropletName = dropletInfo?.name || dropletId;

            if (!token) {
                await bot.editMessageText('❌ Gagal menghapus: Token API tidak ditemukan.', { chat_id: chatId, message_id: cbq.message.message_id });
                return true;
            }

            const result = await doApi.deleteDroplet(token, dropletId);
            
            if (result.success || result.status === 404) {
                if (dropletInfo) {
                    googleSheets.updateRowStatusToDeleted(dropletInfo.ip_address).catch(console.error);
                    delete db[apiKeyName].managed_droplets[dropletId];
                    await writeDb(db);
                }
                 await bot.editMessageText(`✅ RDP/VPS "${dropletName}" telah berhasil dihapus.`, { chat_id: chatId, message_id: cbq.message.message_id });
            } else {
                 await bot.editMessageText(`❌ Gagal menghapus "${dropletName}".\nAlasan: ${result.message}`, { chat_id: chatId, message_id: cbq.message.message_id });
            }
        }
        return true;
    }

    return false;
}

module.exports = {
    showDropletList,
    showFullDropletList,
    handleCallbackQuery
};

