/**
 * Script untuk Bot Telegram yang bertugas mengelola daftar grup (grup.txt).
 * Bot ini hanya akan merespon perintah dari admin yang telah ditentukan.
 * Interaksi dilakukan melalui keyboard permanen untuk kemudahan penggunaan.
 *
 * Dibuat oleh: Gemini
 * Tanggal: 29 Juli 2025
 */

const TelegramBot = require('node-telegram-bot-api');
const fs = require('fs');

// --- KONFIGURASI ---
// Token bot yang Anda berikan
const TOKEN = "8413000626:AAGkVJ4rR8yga9JxiC5vdyaejEwf8dJIJXA";

// Daftar ID admin yang diizinkan menggunakan bot
const ADMIN_IDS = [925063095, 818305655];

// Lokasi file daftar grup
const GRUP_FILE = 'grup.txt';

// Inisialisasi bot
const bot = new TelegramBot(TOKEN, { polling: true });

// Objek untuk melacak status/aksi terakhir dari setiap admin
let userState = {};

console.log("✅ Bot manajemen grup telah aktif...");

// --- DEFINISI KEYBOARD ---
const mainMenuKeyboard = {
    reply_markup: {
        keyboard: [
            [{ text: '➕ Tambah Grup' }, { text: '➖ Hapus Grup' }],
            [{ text: '📋 Lihat Daftar' }]
        ],
        resize_keyboard: true, // Membuat keyboard menyesuaikan ukuran layar
        one_time_keyboard: false // Keyboard tidak hilang setelah ditekan
    }
};

const cancelKeyboard = {
    reply_markup: {
        keyboard: [
            [{ text: '⬅️ Batal' }]
        ],
        resize_keyboard: true,
        one_time_keyboard: false
    }
};


// --- FUNGSI BANTUAN ---

// Fungsi untuk memeriksa apakah pengguna adalah admin
function isAdmin(userId) {
    return ADMIN_IDS.includes(userId);
}

// Fungsi untuk membaca daftar grup dari file
function getGroupList() {
    if (!fs.existsSync(GRUP_FILE)) {
        return [];
    }
    const data = fs.readFileSync(GRUP_FILE, 'utf8');
    return data.split(/\r?\n/).filter(line => line.trim() !== "");
}

// Fungsi untuk menyimpan daftar grup ke file
function saveGroupList(groups) {
    const data = groups.join('\n');
    fs.writeFileSync(GRUP_FILE, data, 'utf8');
}


// --- HANDLER BOT UTAMA ---

// Perintah /start untuk menampilkan menu utama
bot.onText(/\/start/, (msg) => {
    if (!isAdmin(msg.from.id)) return;
    userState[msg.from.id] = null; // Reset status pengguna
    bot.sendMessage(msg.chat.id, "Selamat datang, Admin! Silakan pilih menu di bawah:", mainMenuKeyboard);
});

// Listener utama untuk semua pesan yang masuk
bot.on('message', (msg) => {
    const chatId = msg.chat.id;
    const userId = msg.from.id;
    const text = msg.text;

    // Abaikan pesan jika bukan dari admin
    if (!isAdmin(userId)) return;

    // Jika pengguna sedang dalam proses tertentu (menambah/menghapus)
    const currentState = userState[userId];
    if (currentState) {
        if (text === '⬅️ Batal') {
            userState[userId] = null; // Batalkan aksi
            bot.sendMessage(chatId, "Aksi dibatalkan. Kembali ke menu utama.", mainMenuKeyboard);
            return;
        }

        if (currentState === 'adding_groups') {
            const newGroups = text.split(/\s+/);
            let currentGroups = getGroupList();
            let addedCount = 0;

            newGroups.forEach(group => {
                if (group && !currentGroups.includes(group)) {
                    currentGroups.push(group);
                    addedCount++;
                }
            });

            if (addedCount > 0) {
                saveGroupList(currentGroups);
                bot.sendMessage(chatId, `✅ Berhasil menambahkan ${addedCount} grup baru.`);
            } else {
                bot.sendMessage(chatId, `ℹ️ Tidak ada grup baru yang ditambahkan (mungkin sudah ada dalam daftar).`);
            }
            userState[userId] = null; // Reset status
            bot.sendMessage(chatId, "Kembali ke menu utama.", mainMenuKeyboard);
            return;
        }

        if (currentState === 'removing_groups') {
            const groupsToRemove = text.split(/\s+/);
            let currentGroups = getGroupList();
            const initialCount = currentGroups.length;

            currentGroups = currentGroups.filter(group => !groupsToRemove.includes(group));
            const removedCount = initialCount - currentGroups.length;

            if (removedCount > 0) {
                saveGroupList(currentGroups);
                bot.sendMessage(chatId, `✅ Berhasil menghapus ${removedCount} grup.`);
            } else {
                bot.sendMessage(chatId, `ℹ️ Tidak ada grup yang dihapus (mungkin tidak ditemukan dalam daftar).`);
            }
            userState[userId] = null; // Reset status
            bot.sendMessage(chatId, "Kembali ke menu utama.", mainMenuKeyboard);
            return;
        }
    }

    // Handler untuk tombol-tombol di menu utama
    switch (text) {
        case '➕ Tambah Grup':
            userState[userId] = 'adding_groups';
            bot.sendMessage(chatId, "Silakan kirim ID atau username grup yang ingin ditambahkan.\n\nAnda bisa menambahkan lebih dari satu, pisahkan dengan spasi.", cancelKeyboard);
            break;

        case '➖ Hapus Grup':
            userState[userId] = 'removing_groups';
            bot.sendMessage(chatId, "Silakan kirim ID atau username grup yang ingin dihapus.\n\nAnda bisa menghapus lebih dari satu, pisahkan dengan spasi.", cancelKeyboard);
            break;

        case '📋 Lihat Daftar':
            const currentGroups = getGroupList();
            if (currentGroups.length === 0) {
                bot.sendMessage(chatId, "Daftar grup saat ini kosong.", mainMenuKeyboard);
                return;
            }
            const listMessage = "📋 Berikut adalah daftar grup saat ini:\n\n" +
                                "```\n" +
                                currentGroups.join("\n") +
                                "\n```";
            bot.sendMessage(chatId, listMessage, { parse_mode: "Markdown" });
            break;
    }
});
